#include "view.hpp"
#include "shader_cb.h"
#include "playercore/platform/ps4/VideoFrame.hpp"
#include <framework/framework.h>
#include <toolkit/toolkit.h>
#include <gnmx.h>
#include <cassert>

#define MY_VERTEX_SHADER "/app0/shader_vv.sb"
#define MY_FRAGMENT_SHADER "/app0/shader_p.sb"

#define ON_DBG(STRING, ...) \
    printf("%s> " STRING, __func__, ##__VA_ARGS__)

using namespace sce;
using namespace sce::Gnmx;

void View::draw(Gnmx::GfxContext* context)
{
    std::lock_guard<std::mutex> lock(m_mutex);
    if (m_mode == DisplayMode::Disabled || !m_current) {
        return;
    }

    auto vsConstants = static_cast<VsConstants*>(context->allocateFromCommandBuffer(sizeof(VsConstants), Gnm::kEmbeddedDataAlignment4));
    assert(vsConstants != nullptr);

    uint32_t pitch = m_current->getOutputInfo().framePitch;
    uint32_t height = m_current->getOutputInfo().frameHeight;
    uint32_t width = m_current->getOutputInfo().frameWidth;

    context->setTextures(Gnm::kShaderStagePs, 0, 1, &m_lumaTexture);
    context->setTextures(Gnm::kShaderStagePs, 1, 1, &m_chromaTexture);

    // Adjust for difference in pitch and width and any required cropping
    float widthScale = (pitch > 0) ? (float)width / pitch : 1.0f;

    uint32_t cropLeft = m_current->getPictureInfo().frameCropLeftOffset;
    uint32_t cropRight = m_current->getPictureInfo().frameCropRightOffset;
    uint32_t cropTop = m_current->getPictureInfo().frameCropTopOffset;
    uint32_t cropBottom = m_current->getPictureInfo().frameCropBottomOffset;

    vsConstants->tex_coord_left = (width > 0) ? ((float)cropLeft / width) * widthScale : 0.0f;
    vsConstants->tex_coord_top = (height > 0) ? (float)cropTop / height : 0.0f;
    vsConstants->tex_coord_right = (width > 0) ? (((float)width - cropRight) / width) * widthScale : widthScale;
    vsConstants->tex_coord_bottom = (height > 0) ? ((float)height - cropBottom) / height : 1.0f;

    vsConstants->vertical_split = static_cast<int>(m_mode);

    Gnm::Buffer constBuffer;
    constBuffer.initAsConstantBuffer(vsConstants, sizeof(VsConstants));
    context->setConstantBuffers(Gnm::kShaderStageVs, 0, 1, &constBuffer);

    context->setPrimitiveType(Gnm::kPrimitiveTypeTriStrip);
    context->drawIndexAuto(4);

    m_current->finish();
}

void View::onVideoFrameBuffer(const std::shared_ptr<twitch::VideoFrameBuffer>& frameBuffer)
{
    std::lock_guard<std::mutex> lock(m_mutex);
    m_current = nullptr;
    m_frameBuffer = frameBuffer;
}

void View::onFrameNeedsReleasing(const std::shared_ptr<twitch::VideoFrame>& videoFrame)
{
    if (m_current.get() == videoFrame.get()) {
        flush();
    }
}

void View::flush()
{
    std::lock_guard<std::mutex> lock(m_mutex);
    m_current = nullptr;
}

bool View::update()
{
    std::lock_guard<std::mutex> lock(m_mutex);
    std::shared_ptr<twitch::VideoFrame> frame;
    if (m_mode == DisplayMode::Disabled || !m_frameBuffer || !(frame = m_frameBuffer->getFrame())) {
        return true;
    }

    m_current = frame;

    uint32_t bNum = 1;
    uint32_t pitch = m_current->getOutputInfo().framePitch;
    uint32_t height = m_current->getOutputInfo().frameHeight;
    //uint32_t width = m_current.width;
    void* frameBuffer = m_current->getOutputInfo().pFrameBuffer;

    int32_t retVal = 0;
    sce::Gnm::SizeAlign sz;
    sce::Gnm::TextureSpec texSpec;

    // common setting.
    texSpec.init();
    texSpec.m_tileModeHint = sce::Gnm::kTileModeDisplay_LinearAligned;
    texSpec.m_numFragments = sce::Gnm::kNumFragments1;

    // luma texture settings.
    texSpec.m_width = pitch;
    texSpec.m_height = height;

    texSpec.m_format = sce::Gnm::kDataFormatR8Unorm;

    retVal = m_lumaTexture.init(&texSpec);
    if (retVal) {
        ON_DBG("luma texture init:%d failed.\n", retVal);
        assert(false);
        return false;
    }

    sz = m_lumaTexture.getSizeAlign();

    if (sz.m_size != pitch * height * bNum) {
        ON_DBG("unexpected y_texture size ... sz.m_size:%d\n", sz.m_size);
        assert(false);
        return false;
    }

    auto luma_address = frameBuffer; //nv12_buffer;
    m_lumaTexture.setBaseAddress256ByteBlocks((uint32_t)(reinterpret_cast<uint64_t>(luma_address) >> 8));

    // chroma texture settings.
    texSpec.m_width = pitch / 2;
    texSpec.m_height = height / 2;

    texSpec.m_format = sce::Gnm::kDataFormatR8G8Unorm;

    retVal = m_chromaTexture.init(&texSpec);

    if (retVal) {
        ON_DBG("chroma texture init:%d failed.\n", retVal);
        assert(false);
        return false;
    }

    sz = m_chromaTexture.getSizeAlign();

    if (sz.m_size != pitch * (height / 2) * bNum) {
        ON_DBG("unexpected c_texture size ... sz.m_size:%d\n", sz.m_size);
        assert(false);
        return false;
    }

    auto chroma_address = (uint8_t*)frameBuffer /*nv12_buffer*/ + (pitch * height * bNum);
    m_chromaTexture.setBaseAddress256ByteBlocks((uint32_t)(reinterpret_cast<uint64_t>(chroma_address) >> 8));

    return true;
}
