﻿#include "pch.h"
#include "App.h"

#include <ppltasks.h>

using namespace nativeplayer;

using namespace concurrency;
using namespace Windows::ApplicationModel;
using namespace Windows::ApplicationModel::Core;
using namespace Windows::ApplicationModel::Activation;
using namespace Windows::UI::Core;
using namespace Windows::UI::Input;
using namespace Windows::System;
using namespace Windows::Foundation;
using namespace Windows::Graphics::Display;



// The main function is only used to initialize our IFrameworkView class.
[Platform::MTAThread]
int main(Platform::Array<Platform::String ^> ^)
{
    auto direct3DApplicationSource = ref new Direct3DApplicationSource();
    CoreApplication::Run(direct3DApplicationSource);
    return 0;
}

IFrameworkView ^Direct3DApplicationSource::CreateView()
{
    return ref new App();
}

App::App() :
    m_windowClosed(false),
    m_windowVisible(true)
{
}

// The first method called when the IFrameworkView is being created.
void App::Initialize(CoreApplicationView ^applicationView)
{
    // Register event handlers for app lifecycle. This example includes Activated, so that we
    // can make the CoreWindow active and start rendering on the window.
    applicationView->Activated +=
        ref new TypedEventHandler<CoreApplicationView ^, IActivatedEventArgs ^>(this, &App::OnActivated);

    CoreApplication::Suspending +=
        ref new EventHandler<SuspendingEventArgs ^>(this, &App::OnSuspending);

    CoreApplication::Resuming +=
        ref new EventHandler<Platform::Object ^>(this, &App::OnResuming);

    // At this point we have access to the device.
    // We can create the device-dependent resources.
    m_deviceResources = std::make_shared<DX::DeviceResources>();

    // TODO: pass-in the actual CommandLine
    m_application.parseCommandLine(0, NULL);
    m_application.initialize(m_deviceResources->GetDevice());
}

// Called when the CoreWindow object is created (or re-created).
void App::SetWindow(CoreWindow ^window)
{
    window->SizeChanged +=
        ref new TypedEventHandler<CoreWindow ^, WindowSizeChangedEventArgs ^>(this, &App::OnWindowSizeChanged);

    window->VisibilityChanged +=
        ref new TypedEventHandler<CoreWindow ^, VisibilityChangedEventArgs ^>(this, &App::OnVisibilityChanged);

    window->Closed +=
        ref new TypedEventHandler<CoreWindow ^, CoreWindowEventArgs ^>(this, &App::OnWindowClosed);

    window->PointerPressed +=
        ref new TypedEventHandler<CoreWindow ^, PointerEventArgs ^>(this, &App::OnPointerPressed);

    window->PointerMoved +=
        ref new TypedEventHandler<CoreWindow ^, PointerEventArgs ^>(this, &App::OnPointerMoved);

    window->PointerReleased +=
        ref new TypedEventHandler<CoreWindow ^, PointerEventArgs ^>(this, &App::OnPointerReleased);


    DisplayInformation ^currentDisplayInformation = DisplayInformation::GetForCurrentView();

    currentDisplayInformation->DpiChanged +=
        ref new TypedEventHandler<DisplayInformation ^, Object ^>(this, &App::OnDpiChanged);

    currentDisplayInformation->OrientationChanged +=
        ref new TypedEventHandler<DisplayInformation ^, Object ^>(this, &App::OnOrientationChanged);

    DisplayInformation::DisplayContentsInvalidated +=
        ref new TypedEventHandler<DisplayInformation ^, Object ^>(this, &App::OnDisplayContentsInvalidated);

    window->KeyUp += 
        ref new TypedEventHandler<CoreWindow ^, KeyEventArgs ^>(this, &App::OnKeyUp);

    m_deviceResources->SetWindow(window);
}

// Initializes scene resources, or loads a previously saved app state.
void App::Load(Platform::String ^entryPoint)
{
    if (m_main == nullptr) {
        m_main = std::unique_ptr<nativeplayerMain>(new nativeplayerMain(m_deviceResources, m_application));
    }
}

// This method is called after the window becomes active.
void App::Run()
{
    while (!m_windowClosed) {
        if (m_windowVisible) {
            CoreWindow::GetForCurrentThread()->Dispatcher->ProcessEvents(CoreProcessEventsOption::ProcessAllIfPresent);

            m_application.update();
            m_main->Update();

            if (m_main->Render()) {
                m_deviceResources->Present();
            }
        } else {
            CoreWindow::GetForCurrentThread()->Dispatcher->ProcessEvents(CoreProcessEventsOption::ProcessOneAndAllPending);
        }
    }
}

// Required for IFrameworkView.
// Terminate events do not cause Uninitialize to be called. It will be called if your IFrameworkView
// class is torn down while the app is in the foreground.
void App::Uninitialize()
{
}

// Application lifecycle event handlers.

void App::OnActivated(CoreApplicationView ^applicationView, IActivatedEventArgs ^args)
{
    // Run() won't start until the CoreWindow is activated.
    CoreWindow::GetForCurrentThread()->Activate();
}

void App::OnSuspending(Platform::Object ^sender, SuspendingEventArgs ^args)
{
    // Force MediaPlayer dtor to run from the main-thread, which is what most apps do
    m_application.finalize();

    // Save app state asynchronously after requesting a deferral. Holding a deferral
    // indicates that the application is busy performing suspending operations. Be
    // aware that a deferral may not be held indefinitely. After about five seconds,
    // the app will be forced to exit.
    SuspendingDeferral ^deferral = args->SuspendingOperation->GetDeferral();

    create_task([this, deferral]() {
        m_deviceResources->Trim();
        deferral->Complete();
    });
}

void App::OnResuming(Platform::Object ^sender, Platform::Object ^args)
{
    // Restore any data or state that was unloaded on suspend. By default, data
    // and state are persisted when resuming from suspend. Note that this event
    // does not occur if the app was previously terminated.
}

// Window event handlers.

void App::OnWindowSizeChanged(CoreWindow ^sender, WindowSizeChangedEventArgs ^args)
{
    m_deviceResources->SetLogicalSize(Size(sender->Bounds.Width, sender->Bounds.Height));
    m_main->CreateWindowSizeDependentResources();
}

void App::OnVisibilityChanged(CoreWindow ^sender, VisibilityChangedEventArgs ^args)
{
    m_windowVisible = args->Visible;
}

void App::OnWindowClosed(CoreWindow ^sender, CoreWindowEventArgs ^args)
{
    m_windowClosed = true;
}

void App::OnPointerPressed(CoreWindow ^sender, PointerEventArgs ^e)
{
    // When the pointer is pressed begin tracking the pointer movement.
    m_main->StartTracking();
    m_main->TrackingUpdate(Point());
    m_pointerStartPosition = e->CurrentPoint->Position;

    m_main->OnPointerPressed(e->CurrentPoint->Position);
}

void App::OnPointerMoved(CoreWindow ^sender, PointerEventArgs ^e)
{
    // Update the pointer tracking code.
    if (m_main->IsTracking()) {
        Point diff;
        diff.X = e->CurrentPoint->Position.X - m_pointerStartPosition.X;
        diff.Y = e->CurrentPoint->Position.Y - m_pointerStartPosition.Y;
        m_main->TrackingUpdate(diff);
    }
}

void App::OnPointerReleased(CoreWindow ^sender, PointerEventArgs ^e)
{
    // Stop tracking pointer movement when the pointer is released.
    m_main->StopTracking();

    m_main->OnPointerReleased(e->CurrentPoint->Position);
}

void App::OnKeyUp(Windows::UI::Core::CoreWindow ^ sender, Windows::UI::Core::KeyEventArgs ^ e)
{
    switch (e->VirtualKey)
    {
        case VirtualKey::GamepadA:
        case VirtualKey::A:
            m_application.play();
            break;
        case VirtualKey::GamepadX:
        case VirtualKey::X:
            m_application.pause();
            break;
        case VirtualKey::GamepadY:
        case VirtualKey::Y:
            m_application.selectPreviousVideo();
            break;
        case VirtualKey::GamepadB:
        case VirtualKey::B:
            m_application.selectNextVideo();
            break;
        case VirtualKey::GamepadDPadUp:
        case VirtualKey::Up:
            m_application.seekBackward30Seconds();
            break;
        case VirtualKey::GamepadDPadDown:
        case VirtualKey::Down:
            m_application.seekForward30Seconds();
            break;
        case VirtualKey::GamepadDPadLeft:
        case VirtualKey::Left:
            m_application.seekBackward5Seconds();
            break;
        case VirtualKey::GamepadDPadRight:
        case VirtualKey::Right:
            m_application.seekForward5Seconds();
            break;
        case VirtualKey::GamepadLeftTrigger:
            m_application.setPreviousQuality();
            break;
        case VirtualKey::GamepadRightTrigger:
            m_application.setNextQuality();
            break;
        default:
            return;
    }
    e->Handled = true;
}

// DisplayInformation event handlers.

void App::OnDpiChanged(DisplayInformation ^sender, Object ^args)
{
    // Note: The value for LogicalDpi retrieved here may not match the effective DPI of the app
    // if it is being scaled for high resolution devices. Once the DPI is set on DeviceResources,
    // you should always retrieve it using the GetDpi method.
    // See DeviceResources.cpp for more details.
    m_deviceResources->SetDpi(sender->LogicalDpi);
    m_main->CreateWindowSizeDependentResources();
}

void App::OnOrientationChanged(DisplayInformation ^sender, Object ^args)
{
    m_deviceResources->SetCurrentOrientation(sender->CurrentOrientation);
    m_main->CreateWindowSizeDependentResources();
}

void App::OnDisplayContentsInvalidated(DisplayInformation ^sender, Object ^args)
{
    m_deviceResources->ValidateDevice();
}
