var MediaPlayerPackage = window.MediaPlayerPackage = require('../../platforms/web/js/index');
var MediaPlayer = window.MediaPlayer = MediaPlayerPackage.MediaPlayer;
var PlayerState = window.PlayerState = MediaPlayerPackage.PlayerState;
var PlayerEvent = window.PlayerEvent = MediaPlayerPackage.PlayerEvent;
var RemotePlayerEvent = window.RemoteEvent = MediaPlayerPackage.RemotePlayerEvent;
var Profile = window.Profile = MediaPlayerPackage.Profile;
var MetadataEvent = window.MetadataEvent = MediaPlayerPackage.MetadataEvent;
var ProfilerEvent = window.ProfilerEvent = MediaPlayerPackage.ProfilerEvent;
var ErrorType = window.ErrorType = MediaPlayerPackage.ErrorType;
var getAccessToken = MediaPlayerPackage.getAccessToken;
var createMasterPlaylistUrl = MediaPlayerPackage.createMasterPlaylistUrl;
var KEY_SYSTEMS = require('../../platforms/web/js/drm/constants').KEY_SYSTEMS;

window.Promise = require('../../platforms/web/js/promise').default;

// Parse query parameters
var params = (function () {
    var params = {};
    var query = location.search.substring(1);
    var vars = query.split('&');
    for (var i = 0; i < vars.length; i++) {
        var pair = vars[i].split('=');
        params[decodeURIComponent(pair[0])] = decodeURIComponent(pair[1]);
    }
    return params;
}());

var useWasm = MediaPlayerPackage.isWasmSupported && !params.asmjs;

var player = window.player = (function (scripts) {
    var mediaPlayerSrc;
    for (var i = 0; i < scripts.length; i++) {
        if (/samplemediaplayer(\.min)?\.js/.test(scripts[i].src)) {
            mediaPlayerSrc = scripts[i].src;
            break;
        }
    }
    var workerUrl = mediaPlayerSrc.replace(/[\w-\.]+\.js$/g, useWasm ? WASM_WORKER_URL : ASMJS_WORKER_URL);
    var wasmUrl = mediaPlayerSrc.replace(/[\w-\.]+\.js$/g, WASM_BINARY_URL);
    var worker = MediaPlayer.createWorker(workerUrl, wasmUrl);

    worker.addEventListener('error', function (e) {
        console.error('worker error:', e);
    });
    var keySystem;
    switch (params.cdm) {
        case 'wv':
            keySystem = KEY_SYSTEMS.WIDEVINE.uuid;
            break;
        case 'pr':
            keySystem = KEY_SYSTEMS.PLAYREADY.uuid;
            break;
        case 'fp':
            keySystem = KEY_SYSTEMS.FAIRPLAY.uuid;
            break;
    }

    return new MediaPlayer({
        latencyValue: params.latencyValue,
        settings: params.settings,
        logLevel: params.logLevel || 'debug',
        keySystem: keySystem,
    }, worker);
}(document.scripts));

console.log('PROFILE:', 'player instance ready', performance.now());

player.setAnalyticsSendEvents(true);

const videoElement = player.getHTMLVideoElement();
document.getElementById('videoContainer').appendChild(videoElement);

// Set up seekbar
(!params.profile && function () {
    var seekbarMax = 0;
    var showBuffer = false;
    var requestHdl = 0;
    var seekbar = document.querySelector('#seekbar span');
    var seekbarText = document.querySelector('#seekbar span span');

    function updateSeekBar() {
        var dur = player.getDuration();
        if (dur === 0) {
            seekbarMax = 0;
            showBuffer = false;
        } else if (showBuffer || dur === Infinity) {
            var buf = player.getBufferDuration();
            seekbarMax = (buf + 1 > seekbarMax ? buf + 1 : seekbarMax);
            seekbar.style.width = 100 * (buf / seekbarMax) + '%';
            seekbarText.innerText = buf.toFixed(1);
        } else {
            var pos = player.getPosition();
            seekbar.style.width = 100 * (pos / dur) + '%';
            seekbarText.innerText = pos.toFixed(1);
        }
        requestHdl = requestAnimationFrame(updateSeekBar);
    }

    document.querySelector('#seekbar').addEventListener('click', function (e) {
        var dur = player.getDuration();
        if (dur !== Infinity && dur > 0) {
            if (e.shiftKey) {
                showBuffer = !showBuffer;
            } else {
                player.seekTo((e.offsetX / this.offsetWidth) * dur);
            }
        }
    });
    document.querySelector('#seekbar').addEventListener('contextmenu', function (e) {
        e.preventDefault();
        if (seekbarText.innerText === 'DISABLED') {
            updateSeekBar();
        } else {
            cancelAnimationFrame(requestHdl);
            seekbar.style.width = 0;
            seekbarText.innerText = 'DISABLED';
        }
        return false;
    });

    updateSeekBar();
}());

var urlInput = document.getElementById('urlInput');
urlInput.addEventListener('change', function (e) {
    player.load(e.target.value);
    player.play();
});

var playButton = document.getElementById('play-button');
playButton.addEventListener('click', function () {
    player.play();
});

var pauseButton = document.getElementById('pause-button');
pauseButton.addEventListener('click', function () {
    player.pause();
});

document.getElementById('chromecast-start').addEventListener('click', function () {
    player.startRemotePlayback();
});

document.getElementById('chromecast-stop').addEventListener('click', function () {
    player.endRemotePlayback();
});

var qualitySelect = document.getElementById('qualities');
qualitySelect.addEventListener('change', function (e) {

    if (e.target.value.startsWith("auto")) {
        player.setAutoSwitchQuality(true);
    } else {
        var quality = player.getQualities().find(function (q) {
            return q.name === e.target.value;
        });
        if (quality) {
            player.setAutoSwitchQuality(false);
            player.setQuality(quality);
        }
    }
});

var speeds = document.getElementById('speeds');
speeds.addEventListener('change', function (e) {
    player.setPlaybackRate(parseFloat(e.target.value));
});

var numBufferEmpties = 0;

player.addEventListener(PlayerEvent.INITIALIZED, function (state) {
    console.log('INITIALIZED:', state);
})
player.addEventListener(PlayerEvent.QUALITY_CHANGED, function (state) {
    var qualityName = player.getQuality().name;
    if (player.getAutoSwitchQuality()) {
        qualitySelect.options[0].text = "Auto " + qualityName;
    } else {
        for (var i = 0; i < qualitySelect.options.length; i++) {
            if (qualitySelect.options[i].value === qualityName) {
                qualitySelect.options[i].selected = true;
                break;
            }
        }
    }

    console.log('QUALITY_CHANGED:', state);
})
player.addEventListener(PlayerEvent.AUTO_SWITCH_QUALITY_CHANGED, function (state) {
    console.log('AUTO_SWITCH_QUALITY_CHANGED:', state);
})
player.addEventListener(PlayerEvent.DURATION_CHANGED, function (state) {
    console.log('DURATION_CHANGED:', state);
})
player.addEventListener(PlayerEvent.VOLUME_CHANGED, function () {
    console.log('VOLUME_CHANGED:', player.getVolume());
})
player.addEventListener(PlayerEvent.MUTED_CHANGED, function () {
    console.log('MUTED_CHANGED:', player.isMuted());
    if (player.isMuted()) {
        muteButton.value = 'Unmute';
    } else {
        muteButton.value = 'Mute';
    }
})
player.addEventListener(PlayerEvent.PLAYBACK_RATE_CHANGED, function () {
    console.log('PLAYBACK_RATE_CHANGED:', player.getPlaybackRate());
})
player.addEventListener(PlayerEvent.REBUFFERING, function () {
    numBufferEmpties++;
    console.log('REBUFFERING');
})
player.addEventListener(PlayerEvent.AUDIO_BLOCKED, function () {
    console.warn('AUDIO_BLOCKED. Audio automatically muted')
})
player.addEventListener(PlayerEvent.PLAYBACK_BLOCKED, function () {
    console.warn('PLAYBACK_BLOCKED. Forced to pause')
})
player.addEventListener(PlayerEvent.ERROR, function (err) {
    console.error('ERROR:', err);
})
player.addEventListener(PlayerEvent.TRACKING, function (event) {
    console.warn(Math.round(performance.now()) / 1000, 'TRACKING:', event.name, event.properties);
})
player.addEventListener(PlayerEvent.TIME_UPDATE, function () {
    if (!params.profile) {
        updateInfo(player);
    }
})
player.addEventListener(PlayerEvent.BUFFER_UPDATE, function () {
    if (!params.profile) {
        updateInfo(player);
    }
})
player.addEventListener(PlayerEvent.SEEK_COMPLETED, function () {
    console.log('SEEK_COMPLETED:', player.getPosition());
})
player.addEventListener(PlayerState.IDLE, function (state) {
    console.log('IDLE');
})
player.addEventListener(PlayerState.BUFFERING, function (state) {
    console.log('BUFFERING');
})
player.addEventListener(PlayerState.PLAYING, function (state) {
    console.log('PLAYING');
})
player.addEventListener(PlayerState.ENDED, function (state) {
    console.log('ENDED');
})
player.addEventListener(PlayerState.READY, function () {
    console.log('READY:', player.getSessionData());
})
player.addEventListener(PlayerEvent.METADATA, function (metadata) {
    if (metadata.type === 'text/json') {
        console.log('METADATA:', JSON.parse(metadata.data));
    } else if (metadata.type === 'application/octet-stream') {
        console.log('BINARY METADATA:', metadata.data);
    }
})
player.addEventListener(PlayerEvent.PROFILE, function (event) {
    console.log('PROFILE:', event, performance.now());
});
player.addEventListener(PlayerEvent.WORKER_ERROR, function (event) {
    console.log('WORKER_ERROR:', event, performance.now());
});
player.addEventListener(RemotePlayerEvent.NO_DEVICES_AVAILABLE, function () {
    console.warn('Remote playback is not available');
    document.getElementById('chromecast-start').disabled = true;
});
player.addEventListener(RemotePlayerEvent.AVAILABLE, function () {
    console.warn('Remote playback is available');
    document.getElementById('chromecast-start').disabled = false;
});
player.addEventListener(RemotePlayerEvent.SESSION_STARTED, function (deviceName) {
    console.warn('Remote playback started on ', deviceName);
});
player.addEventListener(RemotePlayerEvent.SESSION_ENDED, function () {
    console.warn('Remote playback ended');
});
player.addEventListener(undefined, function (arg) {
    console.error('Undefined player event emitted. Arg:', arg);
});

// The buttons
var volDownButton = document.getElementById('volume-down');
volDownButton.addEventListener('click', function () {
    player.setVolume(Math.max(0, player.getVolume() - 0.2));
})
var volUpButton = document.getElementById('volume-up');
volUpButton.addEventListener('click', function () {
    player.setVolume(Math.min(1, player.getVolume() + 0.2));
})
var muteButton = document.getElementById('mute');
muteButton.addEventListener('click', function () {
    player.setMuted(!player.isMuted());
})
document.getElementById('seek-random').addEventListener('click', function () {
    var dur = player.getDuration();
    if (dur != Infinity && dur > 0) {
        player.seekTo(Math.random() * player.getDuration())
    }
})

document.addEventListener('keydown', function (event) {
    var dur = player.getDuration();
    if (dur === 0 || dur === Infinity) { return }

    var currentTime = player.getPosition();
    switch (event.keyCode) {
        case 37:
            player.seekTo(currentTime - 5);
            break;
        case 39:
            player.seekTo(currentTime + 5);
            break;
    }
});

if (params.auto) {
    player.setAutoSwitchQuality(params.auto === 'true');
}

if (params.muted === 'true') {
    player.setMuted(true);
}

if (params.volume) {
    player.setVolume(parseFloat(params.volume))
}

player.setAutoViewportSizeEnabled(true);
player.setBufferReplaceEnabled(true);

// Load based on input value
function loadAndPlay(src) {
    player.load(src);
    if (params.autoplay !== 'false') {
        urlInput.value = src;
        player.play();
    } else {
        urlInput.value = 'autoplay disabled';
    }
};
if (params.channel) {
    loadAndPlay('https://twitch.tv/' + params.channel);
} else if (params.video) {
    loadAndPlay('https://twitch.tv/video/v/' + params.video);
} else if (params.url) {
    loadAndPlay(params.url);
} else if (params.clip) {
    loadAndPlay('https://clips.twitch.tv/' + params.clip);
} else if (params.streamId) {
    const config = {
        streamId: params.streamId,
        streamType: /^[0-9]+$/.test(params.streamId) ? 'vod' : 'channel',
        clientId: '7xowrgquc8495rq4w5p79aeiwzxglu8',
        playerType: 'site',
        playerBackend: 'mediaplayer',
        platform: 'web',
    }
    getAccessToken(config).then(function (token) {
        loadAndPlay(createMasterPlaylistUrl(config, token));
    });
} else {
    urlInput.value = 'Enter playlist url here';
}

player.addEventListener(PlayerState.READY, function (state) {
    qualitySelect.options.length = 1;

    var currentQuality = player.getQuality();
    player.getQualities().forEach(function (quality) {
        var option = document.createElement('option');
        option.text = quality.name;
        option.value = quality.name;
        qualitySelect.appendChild(option);

        if (!player.getAutoSwitchQuality() && currentQuality.name === quality.name) {
            option.selected = true;
        }
    });
});

// Get useful information out of a media source
function getMSEStats(video, mediaSource) {
    var result = {};

    result.mediaSourceDuration = mediaSource.duration;
    result.mediaSourceReadyState = mediaSource.readyState;
    result.bufferEmptyEvents = numBufferEmpties;

    var sourceBuffers = mediaSource.sourceBuffers;
    if (sourceBuffers && sourceBuffers.length > 0) {
        var primaryBuffer = sourceBuffers[0];
        result.sourceBufferMode = primaryBuffer.mode;
        result.updating = primaryBuffer.updating;
        var buffered = video.buffered;
        var str = '';
        for (var index = 0; index < buffered.length; index++) {
            str += '<br>start: ' + buffered.start(index).toFixed(3) +
                ' end: ' + buffered.end(index).toFixed(3);
        }
        result.bufferTimeRanges = str;
        result.currentPlayHead = video.currentTime.toFixed(3);
        result.timestampOffset = primaryBuffer.timestampOffset;

        var str = '';
        for (var t = 0; t < video.textTracks.length; t++) {
            str += 'Track ' + t + ':<br>';
            var cues = video.textTracks[t].cues;
            for (var c = 0; c < cues.length; c++) {
                str += c + ': ' + cues[c].startTime.toFixed(2) + '<br>';
            }
        }
        result.activeCues = str;
    }
    return result;
};

function objectAssign() {
    var target = arguments[0];
    for (var index = 1; index < arguments.length; index++) {
        var source = arguments[index];
        for (var prop in source) {
            if (source.hasOwnProperty(prop)) {
                target[prop] = source[prop];
            }
        }
    }
    return target;
}

function throttle(fn) {
    var pending = false;
    function call() {
        pending = false;
        fn();
    }

    return function () {
        if (!pending) {
            pending = true;
            requestAnimationFrame(call);
        }
    }
}

function constructStats(player) {
    var playerInfo = {
        bitrate: (player.getVideoBitRate() / 1000).toFixed(0) + ' kbps',
        framerate: player.getVideoFrameRate().toFixed(0),
        latency: player.getLiveLatency().toFixed(2) + 's',
        paused: player.isPaused(),
        position: player.getPosition().toFixed(2) + 's',
        bufferSize: player.getBufferDuration().toFixed(2) + 's',
        volume: 100 * player.getVolume().toFixed(2) + '%',
        droppedFrames: player.getDroppedFrames() + '/' + player.getDecodedFrames(),
        displayResolution: player.getDisplayWidth() + ' x ' + player.getDisplayHeight(),
        videoResolution: player.getVideoWidth() + ' x ' + player.getVideoHeight(),
        backend: useWasm ? 'WebAssembly' : 'asm.js',
    };
    return objectAssign(playerInfo, player.getManifestInfo());
}

// Populate and update table
var playbackStatsTable = document.getElementById('playbackstats');
var mseStatsTable = document.getElementById('msestats');

// Get a row to the stats table.  If the stat does not exist, create it
//
// @param parent stats table (i.e. playback, network, MSE)
// @param statId id/name of the state to add (to be used as HTML id)
// @param statValue value for the element in the table
function addOrUpdateStat(table, statId, statValue) {
    var stat = document.getElementById(statId);
    if (!stat) {
        // Add keyRow to videoInfo
        var row = document.createElement('tr');

        var label = document.createElement('th');
        label.innerHTML = statId;
        row.appendChild(label);

        stat = document.createElement('td');
        stat.id = statId;
        row.appendChild(stat);
        table.appendChild(row);
    }

    stat.innerHTML = formatStatValue(statValue);
}

function formatStatValue(value) {
    if (typeof value == 'number') {
        // Limit to 3 decimal places
        return Number(value.toFixed(3));
    } else {
        return value;
    }
}

var updateInfo = throttle(function () {
    var playbackStats = constructStats(player);
    for (var key in playbackStats) {
        addOrUpdateStat(playbackStatsTable, key, playbackStats[key]);
    }

    var sink = player.mediaSinkManager.getCurrentSink();
    if (sink && sink.mediaSource) {
        sink.mediaSource.then(function (mediaSource) {
            var mseStats = getMSEStats(player.getHTMLVideoElement(), mediaSource);
            for (var key in mseStats) {
                addOrUpdateStat(mseStatsTable, key, mseStats[key]);
            }
        })
    }
});
