﻿#pragma once

namespace DX {
// Provides an interface for an application that owns DeviceResources to be notified of the device being lost or created.
interface IDeviceNotify {
    virtual void OnDeviceLost() = 0;
    virtual void OnDeviceRestored() = 0;
};

// Controls all the DirectX device resources.
class DeviceResources {
public:
    DeviceResources();
    void SetWindow(WindowRef window);
#ifdef __cplusplus_winrt
    void SetCurrentOrientation(Windows::Graphics::Display::DisplayOrientations currentOrientation);
    void Trim();
#endif
    void SetLogicalSize(Windows::Foundation::Size logicalSize);
    void SetDpi(float dpi);
    void ValidateDevice();
    void HandleDeviceLost();
    void RegisterDeviceNotify(IDeviceNotify *deviceNotify);
    void Present();

    Microsoft::WRL::ComPtr<ID3D11DEVICE> GetDevice() { return m_d3dDevice; }

    // The size of the render target, in pixels.
    Windows::Foundation::Size   GetOutputSize() const { return m_outputSize; }

    // The size of the render target, in dips.
    Windows::Foundation::Size   GetLogicalSize() const { return m_logicalSize; }
    float                       GetDpi() const { return m_effectiveDpi; }

    // D3D Accessors.
    ID3D11DEVICE               *GetD3DDevice() const { return m_d3dDevice.Get(); }
    ID3D11DEVICECONTEXT        *GetD3DDeviceContext() const { return m_d3dContext.Get(); }
    IDXGISWAPCHAIN             *GetSwapChain() const { return m_swapChain.Get(); }
    D3D_FEATURE_LEVEL           GetDeviceFeatureLevel() const { return m_d3dFeatureLevel; }
    ID3D11RENDERTARGETVIEW     *GetBackBufferRenderTargetView() const { return m_d3dRenderTargetView.Get(); }
    ID3D11DepthStencilView     *GetDepthStencilView() const { return m_d3dDepthStencilView.Get(); }
    D3D11_VIEWPORT              GetScreenViewport() const { return m_screenViewport; }
    DirectX::XMFLOAT4X4         GetOrientationTransform3D() const { return m_orientationTransform3D; }

    // D2D Accessors.
    ID2D1FACTORY               *GetD2DFactory() const { return m_d2dFactory.Get(); }
    ID2D1DEVICE                *GetD2DDevice() const { return m_d2dDevice.Get(); }
    ID2D1DEVICECONTEXT         *GetD2DDeviceContext() const { return m_d2dContext.Get(); }
    ID2D1Bitmap1               *GetD2DTargetBitmap() const { return m_d2dTargetBitmap.Get(); }
    IDWRITEFACTORY             *GetDWriteFactory() const { return m_dwriteFactory.Get(); }
    D2D1::Matrix3x2F            GetOrientationTransform2D() const { return m_orientationTransform2D; }

private:
    void CreateDeviceIndependentResources();
    void CreateDeviceResources();
    void CreateWindowSizeDependentResources();
    void UpdateRenderTargetSize();
    DXGI_MODE_ROTATION ComputeDisplayRotation();

    // Direct3D objects.
    Microsoft::WRL::ComPtr<ID3D11DEVICE>            m_d3dDevice;
    Microsoft::WRL::ComPtr<ID3D11DEVICECONTEXT>     m_d3dContext;
    Microsoft::WRL::ComPtr<IDXGISWAPCHAIN>          m_swapChain;

    // Direct3D rendering objects. Required for 3D.
    Microsoft::WRL::ComPtr<ID3D11RENDERTARGETVIEW>  m_d3dRenderTargetView;
    Microsoft::WRL::ComPtr<ID3D11DepthStencilView>  m_d3dDepthStencilView;
    D3D11_VIEWPORT                                  m_screenViewport;

    // Direct2D drawing components.
    Microsoft::WRL::ComPtr<ID2D1FACTORY>        m_d2dFactory;
    Microsoft::WRL::ComPtr<ID2D1DEVICE>         m_d2dDevice;
    Microsoft::WRL::ComPtr<ID2D1DEVICECONTEXT>  m_d2dContext;
    Microsoft::WRL::ComPtr<ID2D1Bitmap1>        m_d2dTargetBitmap;

    // DirectWrite drawing components.
    Microsoft::WRL::ComPtr<IDWRITEFACTORY>      m_dwriteFactory;

    // Cached reference to the Window.
#ifdef __cplusplus_winrt
    Platform::Agile<Windows::UI::Core::CoreWindow> m_window;
#else
    HWND m_window;
#endif

    // Cached device properties.
    D3D_FEATURE_LEVEL                               m_d3dFeatureLevel;
    Windows::Foundation::Size                       m_d3dRenderTargetSize;
    Windows::Foundation::Size                       m_outputSize;
    Windows::Foundation::Size                       m_logicalSize;
#ifdef __cplusplus_winrt
    Windows::Graphics::Display::DisplayOrientations m_nativeOrientation;
    Windows::Graphics::Display::DisplayOrientations m_currentOrientation;
#endif
    float                                           m_dpi;

    // This is the DPI that will be reported back to the app. It takes into account whether the app supports high resolution screens or not.
    float m_effectiveDpi;

    // Transforms used for display orientation.
    D2D1::Matrix3x2F    m_orientationTransform2D;
    DirectX::XMFLOAT4X4 m_orientationTransform3D;

    // The IDeviceNotify can be held directly as it owns the DeviceResources.
    IDeviceNotify *m_deviceNotify;
};
}
