#include "pch.h"
#include "../Common/DirectXHelper.h"
#include "TextureAndSRVNV12.hpp"

TextureAndSRVNV12::TextureAndSRVNV12(std::shared_ptr<DX::DeviceResources> deviceResources, int width, int height)
    : m_deviceResources(deviceResources)
    , m_width(width)
    , m_height(height)
{
    // Create the texture
    D3D11_TEXTURE2D_DESC descNV12;
    descNV12.Width = width;
    descNV12.Height = height;
    descNV12.MipLevels = descNV12.ArraySize = 1;
    descNV12.Format = DXGI_FORMAT_NV12;
    descNV12.SampleDesc.Count = 1;
    descNV12.Usage = D3D11_USAGE_DYNAMIC;
    descNV12.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    descNV12.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    descNV12.MiscFlags = 0;
    descNV12.SampleDesc.Count = 1;
    descNV12.SampleDesc.Quality = 0;

    DX::ThrowIfFailed(
        m_deviceResources->GetD3DDevice()->CreateTexture2D(&descNV12, NULL, &m_texture)
    );

    // Now associate texture with the shader
    D3D11_SHADER_RESOURCE_VIEW_DESC srvY;
    srvY.Format = DXGI_FORMAT_R8_UNORM;
    srvY.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
    srvY.TextureCube.MipLevels = descNV12.MipLevels;
    srvY.TextureCube.MostDetailedMip = 0;

    DX::ThrowIfFailed(
        m_deviceResources->GetD3DDevice()->CreateShaderResourceView(
            m_texture.Get(), &srvY, &m_shaderResourceView[0])
    );

    D3D11_SHADER_RESOURCE_VIEW_DESC srvUV;
    srvUV.Format = DXGI_FORMAT_R8G8_UNORM;
    srvUV.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
    srvUV.TextureCube.MipLevels = descNV12.MipLevels;
    srvUV.TextureCube.MostDetailedMip = 0;

    DX::ThrowIfFailed(
        m_deviceResources->GetD3DDevice()->CreateShaderResourceView(
            m_texture.Get(), &srvUV, &m_shaderResourceView[1])
    );

    D3D11_SAMPLER_DESC samplerDesc = CD3D11_SAMPLER_DESC(CD3D11_DEFAULT());
    samplerDesc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
    samplerDesc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
    DX::ThrowIfFailed(
        m_deviceResources->GetD3DDevice()->CreateSamplerState(&samplerDesc, &m_samplerState)
    );
}

TextureAndSRVNV12::~TextureAndSRVNV12()
{
}

void TextureAndSRVNV12::Reset()
{
    m_texture.Reset();
    m_shaderResourceView[0].Reset();
    m_shaderResourceView[1].Reset();
    m_samplerState.Reset();
}

void TextureAndSRVNV12::Update(Microsoft::WRL::ComPtr<ID3D11Texture2D> texture)
{
    D3D11_TEXTURE2D_DESC textureDescription;
    m_texture.Get()->GetDesc(&textureDescription);

    D3D11_BOX box;
    box.left = 0;
    box.top = 0;
    box.front = 0;
    box.right = textureDescription.Width;
    box.bottom = textureDescription.Height;
    box.back = 1;

    m_deviceResources->GetD3DDeviceContext()->CopySubresourceRegion(m_texture.Get(), 0, 0, 0, 0, texture.Get(), 0, &box);
}

Microsoft::WRL::ComPtr<ID3D11Texture2D> TextureAndSRVNV12::getTexture() const
{
    return m_texture;
}

Microsoft::WRL::ComPtr<ID3D11ShaderResourceView> TextureAndSRVNV12::getSRV(int index) const
{
    assert(index >= 0 && index < 2);
    return m_shaderResourceView[index];
}

Microsoft::WRL::ComPtr<ID3D11SamplerState> TextureAndSRVNV12::getSamplerState() const
{
    return m_samplerState;
}
