﻿#include "pch.h"

#include "Common/DirectXHelper.h"
#include "ControlRenderer.h"

#include <algorithm>
#include <codecvt>
#include <locale>
#include <string>

using namespace nativeplayer;
using namespace Microsoft::WRL;

unsigned int ControlRenderer::ms_count = 0;

ControlRenderer::ControlRenderer(const std::shared_ptr<DX::DeviceResources>& deviceResources, const std::string name, const std::function<void()> onClicked)
    : Renderer(deviceResources)
    , m_onClicked(onClicked)
{
    m_text = stringtowstring(name);

    m_index = ms_count++;

    ZeroMemory(&m_textMetrics, sizeof(DWRITE_TEXT_METRICS));

    // Create device independent resources
    ComPtr<IDWriteTextFormat> textFormat;
    DX::ThrowIfFailed(
        m_deviceResources->GetDWriteFactory()->CreateTextFormat(
            L"Consolas",
            nullptr,
            DWRITE_FONT_WEIGHT_DEMI_BOLD,
            DWRITE_FONT_STYLE_NORMAL,
            DWRITE_FONT_STRETCH_NORMAL,
            10.0f,
            L"en-US",
            &textFormat));

    DX::ThrowIfFailed(
        textFormat.As(&m_textFormat));

    DX::ThrowIfFailed(
        m_textFormat->SetParagraphAlignment(DWRITE_PARAGRAPH_ALIGNMENT_NEAR));

    DX::ThrowIfFailed(
        m_deviceResources->GetD2DFactory()->CreateDrawingStateBlock(&m_stateBlock));

    CreateDeviceDependentResources();	
}

void ControlRenderer::Update(DX::StepTimer const&)
{
    ComPtr<IDWriteTextLayout> textLayout;
    DX::ThrowIfFailed(
        m_deviceResources->GetDWriteFactory()->CreateTextLayout(
            m_text.c_str(),
            (uint32_t)m_text.length(),
            m_textFormat.Get(),
            200.0f, // Max width of the input text.
            50.0f, // Max height of the input text.
            &textLayout));

    DX::ThrowIfFailed(textLayout.As(&m_textLayout));

    DX::ThrowIfFailed(m_textLayout->GetMetrics(&m_textMetrics));
}

void ControlRenderer::Render()
{
    ID2D1DeviceContext* context = m_deviceResources->GetD2DDeviceContext();

    Windows::Foundation::Size logicalSize = m_deviceResources->GetLogicalSize();
    m_position = D2D1::Point2F(static_cast<float>((logicalSize.Width / (ms_count + 2)) * (m_index + 1)), 0.f);

    context->SaveDrawingState(m_stateBlock.Get());
    context->BeginDraw();

    DX::ThrowIfFailed(m_textFormat->SetTextAlignment(DWRITE_TEXT_ALIGNMENT_LEADING));

    context->DrawTextLayout(D2D1::Point2F(m_position.x + 2.0f, m_position.y + 2.f), m_textLayout.Get(), m_background.Get());
    context->DrawTextLayout(m_position, m_textLayout.Get(), m_whiteBrush.Get());

    // Ignore D2DERR_RECREATE_TARGET here. This error indicates that the device
    // is lost. It will be handled during the next call to Present.
    HRESULT hr = context->EndDraw();

    if (hr != D2DERR_RECREATE_TARGET) {
        DX::ThrowIfFailed(hr);
    }

    context->RestoreDrawingState(m_stateBlock.Get());
}

void ControlRenderer::CreateDeviceDependentResources()
{
    DX::ThrowIfFailed(m_deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::White), &m_whiteBrush));
    DX::ThrowIfFailed(m_deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::Black, 0.5f), &m_background));
}

void ControlRenderer::ReleaseDeviceDependentResources()
{
    m_whiteBrush.Reset();
    m_background.Reset();
}

bool ControlRenderer::IsInWindow(Windows::Foundation::Point point)
{
    return (point.X >= m_position.x && point.X < (m_position.x + m_textMetrics.width) && point.Y >= m_position.y && point.Y < (m_position.y + m_textMetrics.height));
}

void ControlRenderer::OnPointerPressed(Windows::Foundation::Point)
{
}

void ControlRenderer::OnPointerReleased(Windows::Foundation::Point)
{
    if (m_onClicked) {
        m_onClicked();
    }
}
