﻿#include "pch.h"
#include "nativeplayer.hpp"

#include "DebugRenderer.h"

#include "Common/DirectXHelper.h"

#include <stdarg.h>  // For va_start, etc.

std::string string_format(const char* fmt, ...)
{
    int size = (static_cast<int>(strlen(fmt))) * 2 + 50;   // Use a rubric appropriate for your code
    std::string str;
    va_list ap;

    while (1) {     // Maximum two passes on a POSIX system...
        str.resize(size);
        va_start(ap, fmt);
        int n = vsnprintf((char *)str.data(), size, fmt, ap);
        va_end(ap);

        if (n > -1 && n < size) {  // Everything worked
            str.resize(n);
            return str;
        }

        if (n > -1) { // Needed size returned
            size = n + 1;    // For null char
        } else {
            size *= 2;    // Guess at a larger size (OS specific)
        }
    }

    return str;
}

using namespace nativeplayer;
using namespace Microsoft::WRL;

// Initializes D2D resources used for text rendering.
DebugRenderer::DebugRenderer(const std::shared_ptr<DX::DeviceResources> &deviceResources, twitch::Application & application)
    : Renderer(deviceResources)
    , m_application(application)
    , m_text(L"")
{
    ZeroMemory(&m_textMetrics, sizeof(DWRITE_TEXT_METRICS));

    // Create device independent resources
    ComPtr<IDWriteTextFormat> textFormat;
    DX::ThrowIfFailed(
        m_deviceResources->GetDWriteFactory()->CreateTextFormat(
            L"Consolas",
            nullptr,
            DWRITE_FONT_WEIGHT_LIGHT,
            DWRITE_FONT_STYLE_NORMAL,
            DWRITE_FONT_STRETCH_NORMAL,
            16.0f,
            L"en-US",
            &textFormat
        )
    );

    DX::ThrowIfFailed(
        textFormat.As(&m_textFormat)
    );

    DX::ThrowIfFailed(
        m_textFormat->SetParagraphAlignment(DWRITE_PARAGRAPH_ALIGNMENT_NEAR)
    );

    DX::ThrowIfFailed(
        m_deviceResources->GetD2DFactory()->CreateDrawingStateBlock(&m_stateBlock)
    );

    CreateDeviceDependentResources();
}

// Updates the text to be displayed.
void DebugRenderer::Update(DX::StepTimer const & /*timer*/)
{
    auto mediaPlayer = m_application.getMediaPlayer();

    bool canSeek = mediaPlayer->isSeekable();
    twitch::MediaTime position = mediaPlayer->getPosition();
    twitch::MediaTime bufferedPosition = mediaPlayer->getBufferedPosition();
    twitch::MediaTime duration = mediaPlayer->getDuration();

    using namespace std::chrono;
    bool isLive = (duration == microseconds::max());

    twitch::Player::State state = mediaPlayer->getState();
    const char *stateStr = twitch::Player::stateToString(state);

    // Display time in seconds because that's easier for human to read
    double positionSec = position.seconds();
    double bufferedPositionSec = bufferedPosition.seconds();
    double durationSec = duration.seconds();

    std::string formattedText;
    formattedText = string_format("State: %s\nCanSeek: %s\nPos/Buf (Len): %.03f/%.03f (%.03f)",
                                  stateStr, canSeek ? "true" : "false",
                                  positionSec, bufferedPositionSec, bufferedPositionSec - positionSec);

    if (isLive) {
        formattedText = string_format("%s\nDuration: <LIVE>", formattedText.c_str());
    } else {
        formattedText = string_format("%s\nDuration: %f", formattedText.c_str(), durationSec);
    }

    const std::string &version = mediaPlayer->getVersion();
    formattedText = string_format("%s\nVersion: %s", formattedText.c_str(), version.c_str());

    m_text = std::wstring(formattedText.begin(), formattedText.end());

    ComPtr<IDWriteTextLayout> textLayout;
    DX::ThrowIfFailed(
        m_deviceResources->GetDWriteFactory()->CreateTextLayout(
            m_text.c_str(),
            (uint32_t)m_text.length(),
            m_textFormat.Get(),
            640.0f, // Max width of the input text.
            50.0f, // Max height of the input text.
            &textLayout
        )
    );

    DX::ThrowIfFailed(
        textLayout.As(&m_textLayout)
    );

    DX::ThrowIfFailed(
        m_textLayout->GetMetrics(&m_textMetrics)
    );
}

// Renders a frame to the screen.
void DebugRenderer::Render()
{
    ID2D1DeviceContext *context = m_deviceResources->GetD2DDeviceContext();
    Windows::Foundation::Size logicalSize = m_deviceResources->GetLogicalSize();

    context->SaveDrawingState(m_stateBlock.Get());
    context->BeginDraw();

    // Position on the bottom right corner
    D2D1::Matrix3x2F screenTranslation = D2D1::Matrix3x2F::Translation(
            logicalSize.Width - m_textMetrics.layoutWidth,
            logicalSize.Height - m_textMetrics.height
                                         );

    context->SetTransform(screenTranslation * m_deviceResources->GetOrientationTransform2D());

    DX::ThrowIfFailed(
        m_textFormat->SetTextAlignment(DWRITE_TEXT_ALIGNMENT_TRAILING)
    );

    context->DrawTextLayout(D2D1::Point2F(1.0f, 1.f), m_textLayout.Get(), m_background.Get());
    context->DrawTextLayout(D2D1::Point2F(0.f, 0.f), m_textLayout.Get(), m_whiteBrush.Get());

    // Ignore D2DERR_RECREATE_TARGET here. This error indicates that the device
    // is lost. It will be handled during the next call to Present.
    HRESULT hr = context->EndDraw();

    if (hr != D2DERR_RECREATE_TARGET) {
        DX::ThrowIfFailed(hr);
    }

    context->RestoreDrawingState(m_stateBlock.Get());
}

void DebugRenderer::CreateDeviceDependentResources()
{
    DX::ThrowIfFailed(m_deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::White), &m_whiteBrush));
    DX::ThrowIfFailed(m_deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::Black, 0.75f), &m_background));
}

void DebugRenderer::ReleaseDeviceDependentResources()
{
    m_whiteBrush.Reset();
    m_background.Reset();
}
