﻿#include "pch.h"
#include "nativeplayer.hpp"
#include "TimelineRenderer.h"
#include "Common/DirectXHelper.h"
#include <algorithm>

// <algorithm> on Windows platform doesn't include std::clamp (?)
namespace std {
template<class T, class Compare>
constexpr const T &clamp(const T &v, const T &lo, const T &hi, Compare comp)
{
    return assert(!comp(hi, lo)),
           comp(v, lo) ? lo : comp(hi, v) ? hi : v;
}

template<class T>
constexpr const T &clamp(const T &v, const T &lo, const T &hi)
{
    return clamp(v, lo, hi, std::less<>());
}
}

using namespace nativeplayer;
using namespace Microsoft::WRL;

// Initializes D2D resources used for text rendering.
TimelineRenderer::TimelineRenderer(const std::shared_ptr<DX::DeviceResources> &deviceResources, twitch::Application & application)
    : Renderer(deviceResources)
    , m_application(application)
{
    // Create device independent resources
    ComPtr<IDWriteTextFormat> textFormat;
    DX::ThrowIfFailed(
        m_deviceResources->GetDWriteFactory()->CreateTextFormat(
            L"Consolas",
            nullptr,
            DWRITE_FONT_WEIGHT_LIGHT,
            DWRITE_FONT_STYLE_NORMAL,
            DWRITE_FONT_STRETCH_NORMAL,
            12.0f,
            L"en-US",
            &textFormat
        )
    );

    DX::ThrowIfFailed(
        textFormat.As(&m_textFormat)
    );

    DX::ThrowIfFailed(
        m_textFormat->SetParagraphAlignment(DWRITE_PARAGRAPH_ALIGNMENT_CENTER)
    );

    DX::ThrowIfFailed(
        m_deviceResources->GetD2DFactory()->CreateDrawingStateBlock(&m_stateBlock)
    );

    CreateDeviceDependentResources();
}

// Updates the text to be displayed.
void TimelineRenderer::Update(DX::StepTimer const & /*timer*/)
{
    Windows::Foundation::Size logicalSize = m_deviceResources->GetLogicalSize();

    // Figure out bounds of the widget
    m_bounds.left = logicalSize.Width * 0.1f;
    m_bounds.top = logicalSize.Height - 100.f;
    m_bounds.right = logicalSize.Width * 0.9f;
    m_bounds.bottom = m_bounds.top + 10.0f;

    // some generic offset to move the time-line to the left to avoid going over the debug display
    const int xOffset = -50;
    m_bounds.left += xOffset;
    m_bounds.right += xOffset;

    // prevent cropping on the left
    if (m_bounds.left < 0) {
        m_bounds.right += 0 - m_bounds.left;
        m_bounds.left += 0 - m_bounds.left;
    }

    // End bounds

    auto mediaPlayer = m_application.getMediaPlayer();
    double durationInSeconds = mediaPlayer->isSeekable() ? mediaPlayer->getDuration().seconds() : 1.0;

    for (unsigned int index = 0; index < TIMELINE_STEP_COUNT; ++index) {

        double seconds = static_cast<double>(index) / static_cast<double>(TIMELINE_STEP_COUNT - 1) * durationInSeconds;
        char data[16];
        std::snprintf(data, sizeof(data), "|\n%02.2f", seconds);
        m_text[index] = stringtowstring(data);

        ComPtr<IDWriteTextLayout> textLayout;
        DX::ThrowIfFailed(m_deviceResources->GetDWriteFactory()->CreateTextLayout(m_text[index].c_str(), (uint32_t)m_text[index].length(), m_textFormat.Get(), TIME_STEP_WIDTH, TIME_STEP_HEIGHT, &textLayout));
        DX::ThrowIfFailed(textLayout.As(&m_textLayout[index]));
    }
}

// Renders a frame to the screen.
void TimelineRenderer::Render()
{
    auto mediaPlayer = m_application.getMediaPlayer();
    if (!mediaPlayer->isSeekable()) {
        return;
    }

    ID2D1DeviceContext *context = m_deviceResources->GetD2DDeviceContext();
    context->SaveDrawingState(m_stateBlock.Get());
    context->BeginDraw();

    context->FillRectangle(m_bounds, m_alphaBrush.Get());

    twitch::MediaTime position = mediaPlayer->getPosition();
    twitch::MediaTime duration = mediaPlayer->getDuration();

    double ratio = duration == twitch::MediaTime::max() ? 1.0f : (position.seconds() / duration.seconds());

    // Draw the current ratio using a circle
    D2D1_ELLIPSE ellipse;
    ellipse.point.y = (m_bounds.bottom + m_bounds.top) / 2;
    ellipse.point.x = (m_bounds.right - m_bounds.left) * static_cast<float>(ratio) + m_bounds.left;
    ellipse.radiusX = 7.5f;
    ellipse.radiusY = 10.0f;
    context->FillEllipse(ellipse, m_yellowBrush.Get());

    DX::ThrowIfFailed(m_textFormat->SetTextAlignment(DWRITE_TEXT_ALIGNMENT_CENTER));

    const float stepWidth = (m_bounds.right - m_bounds.left) / static_cast<float>(TIMELINE_STEP_COUNT - 1);
    for (unsigned int index = 0; index < TIMELINE_STEP_COUNT; ++index) {
        float offsetX = stepWidth * static_cast<float>(index) - TIME_STEP_WIDTH * 0.5f;
        context->DrawTextLayout(D2D1::Point2F(offsetX + m_bounds.left + 1.0f, m_bounds.top + 1.f), m_textLayout[index].Get(), m_background.Get());
        context->DrawTextLayout(D2D1::Point2F(offsetX + m_bounds.left, m_bounds.top), m_textLayout[index].Get(), m_whiteBrush.Get());
    }

    // Ignore D2DERR_RECREATE_TARGET here. This error indicates that the device
    // is lost. It will be handled during the next call to Present.
    HRESULT hr = context->EndDraw();

    if (hr != D2DERR_RECREATE_TARGET) {
        DX::ThrowIfFailed(hr);
    }

    context->RestoreDrawingState(m_stateBlock.Get());
}

void TimelineRenderer::CreateDeviceDependentResources()
{
    DX::ThrowIfFailed(m_deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::White, 0.5f), &m_alphaBrush));
    DX::ThrowIfFailed(m_deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::White), &m_whiteBrush));
    DX::ThrowIfFailed(m_deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::Yellow, 0.75f), &m_yellowBrush));
    DX::ThrowIfFailed(m_deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::Black, 0.75f), &m_background));
}

void TimelineRenderer::ReleaseDeviceDependentResources()
{
    m_alphaBrush.Reset();
    m_whiteBrush.Reset();
    m_yellowBrush.Reset();
    m_background.Reset();
}

bool TimelineRenderer::IsInWindow(Windows::Foundation::Point point)
{
    return (point.X >= m_bounds.left && point.X < m_bounds.right && point.Y >= m_bounds.top && point.Y < m_bounds.bottom);
}

void TimelineRenderer::OnPointerPressed(Windows::Foundation::Point)
{
}

void TimelineRenderer::OnPointerReleased(Windows::Foundation::Point point)
{
    float ratio = (point.X - m_bounds.left) / static_cast<float>(m_bounds.right - m_bounds.left);
    ratio = std::clamp(ratio, 0.f, 1.f);

    auto mediaPlayer = m_application.getMediaPlayer();

    twitch::MediaTime duration = mediaPlayer->getDuration();
    twitch::MediaTime seekTo = duration * ratio;
    mediaPlayer->seekTo(seekTo);
}
