#include "pch.h"
#include <chrono>
#include <iostream>
#include <cassert>

#include "playercore/Log.hpp"
#include "playercore/platform/windows/PlayerCore.hpp"
#include "nativeplayer.hpp"

using namespace twitch;
using namespace twitch::windows;
// Example:
// nativeplayer -stop-after 3000 -seek-to 2600000 vod://321321321
bool Application::parseCommandLine(int argc, const char **argv)
{
    return m_options.parse(argc, argv);
}

void Application::onQualityChanged(const Quality&)
{
}

void Application::onStateChanged(Player::State newState)
{
    m_playerState = newState;
}

void Application::onError(const Error& error)
{
    m_lastError = error.result;
}

int Application::initialize(Microsoft::WRL::ComPtr<ID3D11DEVICE> device)
{
    auto onRenderedFrame = [this](const MediaSampleBuffer & videoSample, int width, int height) {
        if (OnRenderCallback) {
            OnRenderCallback(videoSample, width, height);
        }
    };

    m_player = twitch::windows::PlayerCore::create(*this, onRenderedFrame, device);

    const std::string &url = m_options.getUrls()[0];
    m_player->load(url);

    return 0;
}

void Application::handleStartSeek()
{
    int seekTo = m_options.getSeekTo();

    // Consume 'seekTo' once the state becomes ready.
    if (m_playerState == Player::State::Ready) {
        if (!m_initialSeek && seekTo != 0) {
            m_player->seekTo(MediaTime(seekTo * 1000000));
            m_initialSeek = true;
        }

        if (!m_playerStarted) {
            m_player->play();
            m_playerStarted = true;
        }
    }
}

int Application::update(void)
{
    handleStartSeek();

    // do we want to stop after a given amount of frames ?
    const int numFramesToRun = m_options.getNumFramesToRun();

    if (numFramesToRun != 0 && m_currentFrameCounter++ == numFramesToRun) {
        return 0;
    }

    if (m_lastError != MediaResult::Ok) {
        return -1;
    }

    return 0;
}

void Application::render()
{
    /* render performs in the display thread. */
    std::this_thread::sleep_for(std::chrono::milliseconds(16));
}

int Application::finalize(void)
{
    m_player->pause();
    m_player.reset();
    assert(m_player.use_count() == 0);
    return 0;
}

void Application::selectNextVideo()
{
    m_currentVideoIndex = (m_currentVideoIndex + 1) % m_options.getUrls().size();

    const std::string &url = m_options.getUrls()[m_currentVideoIndex];
    m_player->load(url);
}

void Application::setPreviousQuality()
{
    Quality targetQuality = m_hasTargetQuality ? m_targetQuality : m_player->getQuality();
    auto& qualities = m_player->getQualities();

    auto iter = std::find(qualities.begin(), qualities.end(), targetQuality);

    // give up if you can't find your own quality.
    if (iter == qualities.end()) {
        return;
    }

    // give up if there's only one quality
    if (qualities.size() == 1) {
        return;
    }

    // cycle around or reduce
    if (iter == qualities.begin()) {
        iter = std::prev(qualities.end());
    } else {
        --iter;
    }

    m_player->setQuality(*iter);
    m_hasTargetQuality = true;
    m_targetQuality = *iter;
}

void Application::setNextQuality()
{
    Quality targetQuality = m_hasTargetQuality ? m_targetQuality : m_player->getQuality();
    auto& qualities = m_player->getQualities();

    auto iter = std::find(qualities.begin(), qualities.end(), targetQuality);

    // give up if you can't find your own quality.
    if (iter == qualities.end()) {
        return;
    }

    // give up if there's only one quality
    if (qualities.size() == 1) {
        return;
    }

    // increment and cycle around as needed
    iter++;
    if (iter == qualities.end()) {
        iter = qualities.begin();
    }

    m_player->setQuality(*iter);
    m_hasTargetQuality = true;
    m_targetQuality = *iter;
}

void Application::selectPreviousVideo()
{
    if (--m_currentVideoIndex < 0) {
        m_currentVideoIndex = static_cast<int>(m_options.getUrls().size() - 1);
    }

    const std::string &url = m_options.getUrls()[m_currentVideoIndex];
    m_player->load(url);
}

void Application::play()
{
    m_player->play();
}

void Application::pause()
{
    m_player->pause();
}

void Application::seekForward5Seconds()
{
    double posInSeconds = m_player->getPosition().seconds() + 5.0;
    if (posInSeconds > m_player->getDuration().seconds()) {
        posInSeconds = m_player->getDuration().seconds();
    }

    twitch::MediaTime seekTo(posInSeconds);
    m_player->seekTo(seekTo);
}

void Application::seekForward30Seconds()
{
    double posInSeconds = m_player->getPosition().seconds() + 30.0;
    if (posInSeconds > m_player->getDuration().seconds()) {
        posInSeconds = m_player->getDuration().seconds();
    }

    twitch::MediaTime seekTo(posInSeconds);
    m_player->seekTo(seekTo);
}

void Application::seekBackward5Seconds()
{
    double posInSeconds = m_player->getPosition().seconds() - 5.0;
    if (posInSeconds < 0.0) {
        posInSeconds = 0.0;
    }

    twitch::MediaTime seekTo(posInSeconds);
    m_player->seekTo(seekTo);
}

void Application::seekBackward30Seconds()
{
    double posInSeconds = m_player->getPosition().seconds() - 30.0;
    if (posInSeconds < 0.0) {
        posInSeconds = 0.0;
    }

    twitch::MediaTime seekTo(posInSeconds);
    m_player->seekTo(seekTo);
}
