#pragma once

#include "options.hpp"
#include "playercore/Player.hpp"
#include "playercore/platform/windows/WindowsPlatform.hpp"

class ApplicationBase {
public:
    virtual int initialize(Microsoft::WRL::ComPtr<ID3D11DEVICE> device) = 0;
    virtual int update() = 0;
    virtual void render() = 0;
    virtual int finalize() = 0;

    using RenderCallback = std::function<void(const twitch::MediaSampleBuffer& sample, int width, int height)>;
    RenderCallback OnRenderCallback;

protected:
private:
};

namespace twitch {

class Application : public ApplicationBase, public Player::Listener {
public:
    // Example:
    // nativeplayer -stop-after 3000 -seek-to 2600000 vod://321321321
    bool parseCommandLine(int argc, const char** argv);
    int initialize(Microsoft::WRL::ComPtr<ID3D11DEVICE> device) override;
    int update() override;
    void render() override;
    int finalize() override;

    const std::shared_ptr<twitch::Player> getMediaPlayer() const { return m_player; }
    std::shared_ptr<twitch::Player> getMediaPlayer() { return m_player; }

    void selectNextVideo();
    void selectPreviousVideo();
    void play();
    void pause();
    void seekForward5Seconds();
    void seekForward30Seconds();
    void seekBackward5Seconds();
    void seekBackward30Seconds();
    void setPreviousQuality();
    void setNextQuality();

    bool hasTargetQuality() const { return m_hasTargetQuality; }
    Quality getTargetQuality() const { assert(m_hasTargetQuality); return m_targetQuality; }

private:
    void handleStartSeek();

    void onDurationChanged(MediaTime) override{};
    void onError(const Error&) override;
    void onMetadata(const std::string&, const std::vector<uint8_t>&) override{};
    void onQualityChanged(const Quality&) override;
    void onRebuffering() override{};
    void onRecoverableError(const Error&) override {};
    void onSeekCompleted(MediaTime) override{};
    void onSessionData(const std::map<std::string, std::string>&) override{}
    void onStateChanged(Player::State state) override;
    void onAnalyticsEvent(const std::string&, const std::string&) override{};
    void onPositionChanged(MediaTime) override {};

    std::shared_ptr<twitch::Player> m_player;
    Options m_options;

    MediaResult m_lastError = MediaResult::Ok;

    // next option to be added, a state machine class will be created. :)
    Player::State m_playerState = Player::State::Idle;
    bool m_playerStarted = false;
    bool m_initialSeek = false;

    int m_currentFrameCounter = 0;
    int m_currentVideoIndex = 0;

    // The current target quality, if any
    bool m_hasTargetQuality = false;
    Quality m_targetQuality;
};
}
