﻿#include "pch.h"
#include "nativeplayer.hpp"
#include "nativeplayerMain.h"
#include "Common\DirectXHelper.h"
#include "Content\ControlRenderer.h"
#include "Content\DebugRenderer.h"
#include "Content\QualityButtonRenderer.h"
#include "Content\TimelineRenderer.h"

using namespace nativeplayer;
using namespace Concurrency;

// Loads and initializes application assets when the application is loaded.
nativeplayerMain::nativeplayerMain(const std::shared_ptr<DX::DeviceResources> &deviceResources, twitch::Application & application) :
    m_deviceResources(deviceResources)
{
    // Register to be notified if the Device is lost or recreated
    m_deviceResources->RegisterDeviceNotify(this);

    m_sceneRenderer = std::make_unique<Sample3DSceneRenderer>(m_deviceResources, application);

    m_renderers.push_back(std::make_shared<DebugRenderer>(m_deviceResources, application));
    m_renderers.push_back(std::make_shared<TimelineRenderer>(m_deviceResources, application));

    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[ <<| ]\n   Y", std::bind(&twitch::Application::selectPreviousVideo, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[ <<-30 ]\n    Up", std::bind(&twitch::Application::seekBackward30Seconds, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[ <<-5 ]\n  Left", std::bind(&twitch::Application::seekBackward5Seconds, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[ Play ]\n   A", std::bind(&twitch::Application::play, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[ Pause ]\n    X", std::bind(&twitch::Application::pause, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[ >>+5 ]\n  Right", std::bind(&twitch::Application::seekForward5Seconds, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[ >>+30 ]\n   Down", std::bind(&twitch::Application::seekForward30Seconds, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[ >>| ]\n   B", std::bind(&twitch::Application::selectNextVideo, &application)));
#ifdef __cplusplus_winrt
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[Q-   ]\n   LT", std::bind(&twitch::Application::setPreviousQuality, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[Q+   ]\n   RT", std::bind(&twitch::Application::setNextQuality, &application)));
#else
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[Q-   ]\n   [", std::bind(&twitch::Application::setPreviousQuality, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(m_deviceResources, "[Q+   ]\n   ]", std::bind(&twitch::Application::setNextQuality, &application)));
#endif

    m_renderers.push_back(std::make_shared<QualityButtonRenderer>(m_deviceResources, application));
}

nativeplayerMain::~nativeplayerMain()
{
    // De-register device notification
    m_deviceResources->RegisterDeviceNotify(nullptr);
}

// Updates application state when the window size changes (e.g. device orientation change)
void nativeplayerMain::CreateWindowSizeDependentResources()
{
    m_sceneRenderer->CreateWindowSizeDependentResources();
}

// Updates the application state once per frame.
void nativeplayerMain::Update()
{
    ProcessInput();

    // Update scene objects.
    m_timer.Tick([&]() {
        m_sceneRenderer->Update(m_timer);

        for (auto & renderer : m_renderers) {
            renderer->Update(m_timer);
        }
    });
}

void nativeplayerMain::OnPointerPressed(Windows::Foundation::Point point)
{
    for (auto & renderer : m_renderers) {
        if (renderer->IsInWindow(point)) {
            renderer->OnPointerPressed(point);
            break;
        }
    }
}

void nativeplayerMain::OnPointerReleased(Windows::Foundation::Point point)
{
    for (auto & renderer : m_renderers) {
        if (renderer->IsInWindow(point)) {
            renderer->OnPointerReleased(point);
            break;
        }
    }
}

// Process all input from the user before updating game state
void nativeplayerMain::ProcessInput()
{
    // TODO: Add per frame input handling here.
    m_sceneRenderer->TrackingUpdate(m_pointerLocation);
}


// Renders the current frame according to the current application state.
// Returns true if the frame was rendered and is ready to be displayed.
bool nativeplayerMain::Render()
{
    // Don't try to render anything before the first Update.
    if (m_timer.GetFrameCount() == 0) {
        return false;
    }

    auto context = m_deviceResources->GetD3DDeviceContext();

    // Reset the viewport to target the whole screen.
    auto viewport = m_deviceResources->GetScreenViewport();
    context->RSSetViewports(1, &viewport);

    // Reset render targets to the screen.
    ID3D11RenderTargetView *const targets[1] = { m_deviceResources->GetBackBufferRenderTargetView() };
    context->OMSetRenderTargets(1, targets, m_deviceResources->GetDepthStencilView());

    // Clear the back buffer and depth stencil view.
    ID3D11RenderTargetView *targetView = m_deviceResources->GetBackBufferRenderTargetView();
    context->ClearRenderTargetView(targetView, DirectX::Colors::CornflowerBlue);
    context->ClearDepthStencilView(m_deviceResources->GetDepthStencilView(), D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 1.0f, 0);

    // Render the scene objects.
    m_sceneRenderer->Render();

    for (auto & renderer : m_renderers) {
        renderer->Render();
    }

    return true;
}

// Notifies renderers that device resources need to be released.
void nativeplayerMain::OnDeviceLost()
{
    m_sceneRenderer->ReleaseDeviceDependentResources();

    for (auto & renderer : m_renderers) {
        renderer->ReleaseDeviceDependentResources();
    }
}

// Notifies renderers that device resources may now be recreated.
void nativeplayerMain::OnDeviceRestored()
{
    m_sceneRenderer->CreateDeviceDependentResources();

    for (auto & renderer : m_renderers) {
        renderer->CreateDeviceDependentResources();
    }

    CreateWindowSizeDependentResources();
}
