#pragma once

#include <algorithm>
#include <cstdarg>
#include <locale>
#include <memory>
#include <string>

namespace twitch {
/** Logging interface */
class Log {
public:
    virtual ~Log() = default;

    /** Logging level enumeration. */
    enum class Level {
        Debug,
        Info,
        Warning,
        Error,
    };

    /**
     *  Logs a formatted string debug message.
     *  @param format format string
     */
    template <typename... Args>
    inline void debug(const std::string& format, Args&&... args) const { log(Level::Debug, format.c_str(), args...); };

    /**
     *  Logs a formatted string error message.
     *  @param format format string
     */
    template <typename... Args>
    inline void error(const std::string& format, Args&&... args) const { log(Level::Error, format.c_str(), args...); };

    /**
     *  Logs a formatted string informational message.
     *  @param format format string
     */
    template <typename... Args>
    inline void info(const std::string& format, Args&&... args) const { log(Level::Info, format.c_str(), args...); };

    /**
     *  Logs a formatted string warning message.
     *  @param format format string
     */
    template <typename... Args>
    inline void warn(const std::string& format, Args&&... args) const { log(Level::Warning, format.c_str(), args...); };

    /**
     *  Logs a formatted string to the log output at the given logging level.
     *  @param level  log level
     *  @param format format string
     */
    void log(Level level, const char* format, ...) const
    {
        va_list args;
        va_start(args, format);
        log(level, format, args);
        va_end(args);
    }

    /**
     *  Logs a formatted string to the log output at the given logging level.
     *  @param level  log level
     *  @param format format string
     *  @param args   format string arguments
     */
    virtual void log(Level level, const char* format, va_list args) const = 0;

    /**
     * Sets the logging level filtering out statements less severe than the given level
     * @param level to set
     */
    virtual void setLevel(Level level) = 0;

    /**
     * Convert human readable string to log level enum
     * @param level to convert
     */
    static inline Level levelFromString(std::string level)
    {
        auto lower = [](char c) { return std::tolower(c, std::locale()); };
        std::transform(level.begin(), level.end(), level.begin(), lower);
        if (level == "debug") {
            return Log::Level::Debug;
        } else if (level == "info") {
            return Log::Level::Info;
        } else if (level == "error") {
            return Log::Level::Error;
        } else {
            return Log::Level::Warning;
        }
    }
};
}
