#pragma once

#include "playercore/MediaFormat.hpp"
#include "playercore/MediaResult.hpp"
#include "playercore/MediaSample.hpp"
#include "playercore/MediaType.hpp"
#include <memory>

namespace twitch {
/**
 * Media decoder interface. Decoders are configured with a MediaFormat. Then given MediaSample
 * instances to decode. When the decoder outputs a buffer hasOutput() should be true. The decoder
 * will then be drained using getOutput() until no more output samples are available.
 */
class MediaDecoder {
public:
    virtual ~MediaDecoder() = default;

    /**
     * Configures the decoder with the given MediaFormat. Configuration occurs before decode().
     *
     * @param input configuration data for the format to decode
     * @param output format of the samples retrieved from getOutput
     * @return MediaResult indicating success or error of the operation
     */
    virtual MediaResult configure(const MediaFormat& input, MediaFormat& output) = 0;

    /**
     * Provides a sample to be decoded.
     * @param input sample to decode
     * @return MediaResult indicating success or error of the operation.
     */
    virtual MediaResult decode(const MediaSampleBuffer& input) = 0;

    /**
     * Indicates a sample discontinuity. Any pending output samples should be flushed. This will be
     * followed with a call to hasOutput() and getOutput() to collect the output samples.
     * @return MediaResult indicating success or error of the operation
     */
    virtual MediaResult flush() = 0;

    /**
     * Resets the decoder instance discarding any output samples. The decoder will then be
     * reconfigured with a new format.
     * @return MediaResult indicating success or error of the operation
     */
    virtual MediaResult reset() = 0;

    /**
     * Checks if the decoder has output samples.
     * @param output true indicating there is more output to process via getOutput(), false
     * otherwise
     * @return MediaResult indicating success or error of the operation.
     */
    virtual MediaResult hasOutput(bool& hasOutput) = 0;

    /**
     * Retrieves the decoder output samples. This method is called after decode() and flush()
     * until hasOutput() returns that no more output is available.
     * @param output decoded sample instance
     * @return MediaResult indicating success or error of the operation.
     */
    virtual MediaResult getOutput(std::shared_ptr<MediaSample>& output) = 0;
};
}
