#pragma once

#include "playercore/MediaType.hpp"
#include <cstdint>
#include <map>
#include <string>
#include <vector>

namespace twitch {
/** Describes the format of a media stream */
class MediaFormat {
public:
    virtual ~MediaFormat() = default;

    /** MediaFormat attributes */
    enum Attribute {
        Video_Width, /** Video width in pixels */
        Video_Height, /** Video height in pixels */
        Video_AVC_NAL_LengthSize, /** AVC NAL length in bytes e.g. 3 or 4 */
        Video_AVC_Level, /** AVC Codec Profile Level */
        Video_AVC_Profile, /** AVC Codec Profile */
        Audio_BitSize, /** PCM audio bit size  e.g. 8 bit or 16 bit */
        Audio_SampleRate, /** Samples per second */
        Audio_SampleSize, /** Max sample size in bytes */
        Audio_ChannelCount, /** Number of audio channels */
    };

    /** MediaFormat codec configuration data keys */
    enum CodecData {
        Video_AVC_SPS, /** AVC Sequence Parameter Set */
        Video_AVC_PPS, /** AVC Picture Parameter Set */
        Video_AVC_AVCC, /** AVCC configuration data */
        Audio_AAC_ESDS, /** AAC ESDS Decoder specific description tag */
    };

    /** @returns User describable name for the format */
    virtual const std::string& getName() const = 0;

    /** @returns Source from which this format originated */
    virtual const std::string& getPath() const = 0;

    /** @returns MediaFormat format type */
    virtual const MediaType& getType() const = 0;

    /**
     *  Checks if the format has the codec specific data for the given type
     *  @param key to check
     *  @return true if the format contains the data for the key false otherwise
     */
    virtual bool hasCodecData(CodecData key) const = 0;

    /**
     *  Gets a byte vector of the code specific data.
     *  @param key to use
     *  @return binary data for the given type
     */
    virtual const std::vector<uint8_t>& getCodecData(CodecData key) const = 0;

    /**
     * For protected content a vector of all protection scheme data supported by the content.
     * For MP4 sourced content will be a vector of PSSH data.
     * @return zero or more binary protection data blobs
     */
    virtual const std::vector<std::vector<uint8_t>>& getProtectionData() const = 0;

    /**
     *  Gets the value of a specific attribute.
     *  @param key to get
     *  @return integer value of the key.
     */
    virtual int getInt(Attribute key) const = 0;

    /**
     *  Checks if the format contains the given attribute.
     *  @param key to check
     *  @return true if the format contains the attribute, false otherwise
     */
    virtual bool hasInt(Attribute key) const = 0;

    /** @return all codec data as a map  */
    virtual const std::map<CodecData, std::vector<uint8_t>>& getCodecData() const = 0;

    /** @return all format data as a map  */
    virtual const std::map<Attribute, int>& getValues() const = 0;

    /**
     *  Sets the name of this format
     *  @param name to set
     */
    virtual void setName(const std::string& name) = 0;

    /**
     *  Sets the source media path identifier of this format
     *  @param source to set
     */
    virtual void setPath(const std::string& source) = 0;

    /**
     *  Sets the type of this format
     *  @param type to set
     */
    virtual void setType(const MediaType& type) = 0;

    /**
     *  Gets a byte vector of the code specific data.
     *  @param key to set
     *  @param data codec specific data to set
     */
    virtual void setCodecData(CodecData key, const std::vector<uint8_t>& data) = 0;

    /**
     *  Sets a format int key/value
     *  @param key to set
     *  @param value to set
     */
    virtual void setInt(Attribute key, int value) = 0;

    /**
     *  Sets all the codec data from a map
     *  @param value to set
     */
    virtual void setCodecData(const std::map<CodecData, std::vector<uint8_t>>& data) = 0;

    /**
     * Sets the protection data
     * @param data zero or more binary protection data blobs
     */
    virtual void setProtectionData(const std::vector<std::vector<uint8_t>>& data) = 0;

    /**
     *  Sets all the format attributes from a map
     *  @param value to set
     */
    virtual void setValues(const std::map<Attribute, int>& values) = 0;
};
}
