#pragma once

#include "playercore/MediaFormat.hpp"
#include "playercore/MediaResult.hpp"
#include "playercore/MediaSample.hpp"
#include "playercore/MediaTime.hpp"
#include <memory>

namespace twitch {
/**
 * Media renderer interface. Renderers are configured with a MediaFormat then given MediaSample
 * instances to render. The render implementation must queue the rendered samples to be output at
 * the sample presentationTime using a give reference clock from Platform::createRenderer().
 */
class MediaRenderer {
public:
    virtual ~MediaRenderer() = default;

    /**
     * Indicates a samples discontinuity any samples queued for rendering should be discarded.
     * @return MediaResult indicating success or error of the operation
     */
    virtual MediaResult flush() = 0;

    /**
     * Starts the instance, if successful render() can be called with media samples of
     * the configured format, if render() samples have been queued in the stopped state then those
     * should be played out.
     *
     * @return MediaResult indicating success or error of the operation
     */
    virtual MediaResult start() = 0;

    /**
     * Stops the renderer instance.
     *
     * @return MediaResult indicating success or error of the operation
     * @note Once stopped, the instance can be restarted with another start() call, or reconfigured
     * via configure() and then another start() call.
     */
    virtual MediaResult stop() = 0;

    /**
     *  Configures the renderer with the given MediaFormat. Configuration occurs before start().
     *
     *  @param format configuration data for the format to render
     *  @return MediaResult indicating success or error of the operation
     */
    virtual MediaResult configure(const MediaFormat& format) = 0;

    /**
     * Provides a sample to be rendered. If the renderer is in the stopped state the sample should
     * be used as a priming sample. For video that means rendering the frame for display (e.g.
     * seeking in video). For audio it is used to prime the audio buffer before start() is called.
     *
     * @param input sample to render
     * @return MediaResult indicating success or error of the operation.
     */
    virtual MediaResult render(std::shared_ptr<const MediaSample> input) = 0;

    /**
     * Gets the most recently presented sample time from the renderer. If the renderer is in the
     * process of rendering a sample and has no current time then MediaTime::invalid() should be
     * returned.
     *
     * @param time timestamp of the mostly recently presented sample including any latency
     * between the render() call and actual presentation time. This value is used to synchronize
     * audio and video playback.
     * @return MediaResult indicating success or error of the operation.
     */
    virtual MediaResult getRenderedPresentationTime(MediaTime& time) = 0;

    /**
     * Sets the current playback rate for samples to be rendered. Input sample presentation times
     * should be scaled to this rate.
     *
     * @param rate to set
     * @return MediaResult indicating success or error of the operation.
     */
    virtual MediaResult setPlaybackRate(float rate) = 0;
};

/**
 * Audio renderer interface extending MediaRenderer with additional methods specifically for audio
 * playback.
 */
class AudioRenderer : public MediaRenderer {
public:
    ~AudioRenderer() override = default;
    /**
     * Sets the playback volume
     * @param volume value to set between 0.0f and 1.0f
     * @return MediaResult indicating success or error of the operation.
     */
    virtual MediaResult setVolume(float volume) = 0;
};

}
