#pragma once

#include <cstdint>
#include <string>

namespace twitch {
/** Return status codes for media player and decoder interfaces */
struct MediaResult {
    int value = 0;
    int code = 0; /** User specified value (platform error code) */

    // filled in with internal codes matching the constants in result.cpp file
    /** Success */
    static const MediaResult Ok;

    /** Generic error occurred */
    static const MediaResult Error;

    /** Method or feature not supported */
    static const MediaResult ErrorNotSupported;

    /** No source present for the operation */
    static const MediaResult ErrorNoSource;

    /** Data or input is invalid for the operation */
    static const MediaResult ErrorInvalidData;

    /** Class or object is an invalid state */
    static const MediaResult ErrorInvalidState;

    /** Method parameter is invalid */
    static const MediaResult ErrorInvalidParameter;

    /** Method or operation timed out */
    static const MediaResult ErrorTimeout;

    /** Output is invalid for the operation */
    static const MediaResult ErrorInvalidOutput;

    /** An unexpected internal error has occurred. */
    static const MediaResult ErrorInternal;

    /** Unspecified Network error */
    static const MediaResult ErrorNetwork;

    /** Error indicating a network I/O failure */
    static const MediaResult ErrorNetworkIO;

    /** Error indicating a network resource is not authorized */
    static const MediaResult ErrorAuthorization;

    /** Error indicating a network resource is not available */
    static const MediaResult ErrorNotAvailable;

    /** only the codes have to match */
    bool operator==(const MediaResult& other) const { return value == other.value; }

    operator int() const { return value; }

    /** can only create from one of the base types */
    constexpr MediaResult(const MediaResult& base, int code) noexcept
        : value(base.value)
        , code(code) {};

    explicit constexpr MediaResult(int value, int code = 0) noexcept
        : value(value)
        , code(code) {};

    MediaResult() noexcept
        : value(Ok)
        , code(0)
    {
    }
};

/**
 * @param result to get the string representation of
 * @return string constant for the given result
 */
const char* mediaResultString(const MediaResult& result);

/** Return custom error codes */
enum class AuthorizationError : int {
    /** Error indicating premium content is not avaliable in your region **/
    ErrorGeoblocked = 1,

    /** Error indicating device does not support playback of premium content **/
    ErrorUnsupportedDevice,

    /** Error indicating a proxy or unblocker has been detected **/
    ErrorAnonymizerBlocked,

    /** Error indicating wi-fi network is needed to watch premium conntent **/
    ErrorCellularNetworkProhibited,

    /** Error indicating restricted premium content **/
    ErrorUnauthorizedEntitlements,

    /** Error indicating restricted vod **/
    ErrorVodRestricted,
};

/**
 * @param result to get the string representation of
 * @return string constant for the given result
 */
const char* authorizationErrorString(const AuthorizationError& error);
}
