#pragma once

#include "playercore/Error.hpp"
#include "playercore/MediaFormat.hpp"
#include "playercore/MediaSample.hpp"
#include "playercore/MediaTime.hpp"
#include "playercore/Statistics.hpp"
#include "playercore/TimeRange.hpp"
#include <memory>

namespace twitch {
/**
 * A MediaSink receives parsed MediaSample instances from an upstream MediaSource. The sink is
 * responsible for buffering sample data and playing sample data out. The sink must signal events
 * back to the owning player instance via MediaSink::Listener interface. This acts as high level
 * playback abstraction where the playback implementation may vary greatly between platforms.
 */
class MediaSink {
public:
    virtual ~MediaSink() = default;

    /**
     * Configures the sink with a new format. The format will contain updated codec information
     * for adaptive streaming.
     * @param track id of the track
     * @param format to configure with
     */
    virtual void configure(int track, std::shared_ptr<MediaFormat> format) = 0;

    /**
     * Indicates the last sample for the current presentation has been appended. The sink can be
     * reusued if flush is called and new data is appended.
     */
    virtual void endOfStream() = 0;

    /**
     * Enqueues a sample for playback
     * @param track id of the track
     * @param sample to add
     */
    virtual void enqueue(int track, std::shared_ptr<MediaSampleBuffer> sample) = 0;

    /**
     * Flushes data for playback.
     */
    virtual void flush() = 0;

    /**
     * Starts playback.
     */
    virtual void play() = 0;

    /**
     * Prepares for playback. This is called when enough samples have been queued to start playback.
     * This method is generally used to prime renderers for playback before actually starting them,
     * e.g. decode keyframe or prime audio buffers.
     */
    virtual void prepare() = 0;

    /**
     * Pauses playback.
     */
    virtual void pause() = 0;

    /**
     * Removes a sample range from the buffered data.
     * @param range to remove
     */
    virtual void remove(const TimeRange& range) = 0;

    /**
     * Rests the sink (removes) all sample data for playback, allowing reuse for another stream or
     * content.
     */
    virtual void reset() = 0;

    /**
     * Seeks to the given position
     * @param time to seek
     */
    virtual void seekTo(MediaTime time) = 0;

    /**
     * Sets the playback rate
     * @param rate to set
     */
    virtual void setPlaybackRate(float rate) = 0;

    /**
     * Sets the video surface to render video to. The type of pointer is platform dependent.
     * @param surface handle to a surface object to set.
     */
    virtual void setSurface(void* surface) = 0;

    /**
     * Sets the playback volume (audio sink only)
     * @param volume
     */
    virtual void setVolume(float volume) = 0;

    enum class State {
        Playing,
        Idle,
        Ended,
    };

    /** Callback interface used to signal events back to the source */
    class Listener {
    public:
        virtual ~Listener() = default;

        /**
         * Invoked when the duration of the sink's source has changed. This method is used only
         * for passthrough mode.
         *
         * @param time duration of the source
         */
        virtual void onSinkDurationChanged(MediaTime duration) = 0;

        /**
         * Invoked when the sink encounters a playback error (e.g. decode failure)
         * @param error description
         */
        virtual void onSinkError(const Error& error) = 0;

        /**
         * Indicates the playing format has changed to the give format
         * @param type content type
         * @param data content
         */
        virtual void onSinkFormatChanged(const MediaFormat& format) = 0;

        /**
         * For tracks with embedded timed metadata sends the sample back to the player for handling
         * @param sample metadata sample
         */
        virtual void onSinkMetadataSample(const MediaSampleBuffer& sample) = 0;

        /**
         * Invoked when the sink encounters a playback error that maybe recovered.
         * @param error description
         */
        virtual void onSinkRecoverableError(const Error& error) = 0;

        /**
         * Indicates sink is in the idle state (waiting for samples or has played all samples)
         * @param new sink state, playing indicates the sink is playing buffered samples,
         * idle state indicates (waiting for samples or has played all samples) and ended indicates
         * the end of the source for passhthrough mode.
         */
        virtual void onSinkStateChanged(State state) = 0;

        /**
         * Updates the current play position (generally only one track should update this value)
         * @param time the current playback time in absolute stream time
         */
        virtual void onSinkTimeUpdate(MediaTime time) = 0;

        /**
         * For the video track the current video playback statistics
         * @param statistics current video statistics
         */
        virtual void onSinkVideoStatistics(const Statistics& statistics) = 0;
    };
};
}
