#pragma once

#include "playercore/Error.hpp"
#include "playercore/MediaFormat.hpp"
#include "playercore/MediaSample.hpp"
#include "playercore/Quality.hpp"
#include "playercore/TimeRange.hpp"
#include <map>
#include <memory>

namespace twitch {
/**
 * Represents a source of data for a Player instance, the source data be retrieved from the network
 * or from a file. A MediaSource must be in the open state to be used, this is done by calling the
 * open() function. The source then signals back to the caller it's ready to be used through the
 * Listener::onSourceOpened callback.
 *
 * A reader of a MediaSource must continuously call read() on the source after it's opened to
 * receive data from it via the Listener callback methods. Once the reader is done reading data
 * it should close() the source. A source can be closed and reopened multiple times possibly
 * retaining some state.
 */
class MediaSource {
public:
    virtual ~MediaSource() = default;

    /**
     * Opens the source for reading.
     */
    virtual void open() = 0;

    /**
     * Closes the source read() will no longer be called.
     */
    virtual void close() = 0;

    /**
     * Seek to given point in the source.
     * @param time to seek to
     */
    virtual void seekTo(MediaTime time) = 0;

    /**
     * Reads samples from the source for the given time period. This method can be asynchronous
     * invoking the Listener methods when ready.
     * @param range range where the start is the player's buffered position and the end is
     *              the desired minimum buffer region to fill
     */
    virtual void read(const TimeRange& range) = 0;

    /** @return true if the source is a live source, false otherwise */
    virtual bool isLive() const = 0;

    /** @return true if the source is in passthrough mode */
    virtual bool isPassthrough() const = 0;

    /** @return true if seeking between the start and duration is supported, false otherwise */
    virtual bool isSeekable() const = 0;

    /** @return Total duration of the stream or MediaTime::max() for an infinite stream  */
    virtual MediaTime getDuration() const = 0;

    /**
     * For network or remote sources sets the timeout for the read() method, if the timeout expires
     * the source should enter an error state. This value can be used to set timeouts on network
     * requests.
     * @param time timeout to set
     */
    virtual void setReadTimeout(MediaTime time) = 0;

    /** @return the current quality of the source */
    virtual const Quality& getQuality() const = 0;

    /**
     * Sets the current quality of the stream, if auto switch quality it is disabled
     * @param quality a valid quality entry from getQualities()
     * @param adaptive true if this an adaptive quality switch, false otherwise, false generally
     * indicates an abrupt switching in which existing requests should be cancelled.
     */
    virtual void setQuality(const Quality& quality, bool adaptive) = 0;

    /**
     * Enables or disables low latency support on the source, if true low latency MAY be used if
     * the source supports it, false if it should not be used.
     * @param enable true to enable, false to disable
     */
    virtual void setLowLatencyEnabled(bool enable) = 0;

    /** @return the available qualities or empty if none are currently available */
    virtual const std::vector<Quality>& getQualities() const = 0;

    /** Optional listener interface for for network request events */
    class Request {
    public:
        virtual ~Request() = default;

        /** @return a string identifying the type of the request */
        virtual const std::string& getType() const = 0;

        /** @return id unique id of the request */
        virtual int getId() const = 0;

        /** @return content length of the data or 0 if empty not unknown */
        virtual size_t getContentLength() const = 0;

        /** @return bitrate of the media if known, 0 otherwise  */
        virtual int getMediaBitrate() const = 0;

        /**  @return duration of the media  */
        virtual MediaTime getMediaDuration() const = 0;

        /** @return true if the media is in flight (prefetch)  */
        virtual bool isMediaPrefetch() const = 0;

        /** @return true if this is a partial request for media  */
        virtual bool isRangeRequest() const = 0;

        /** Listener interface for for request-response progress events */
        class Listener {
        public:
            virtual ~Listener() = default;

            /**
             * Called when a request is sent
             * @param request that is sending
             */
            virtual void onRequestSent(const Request& request) = 0;

            /**
             * Called when the response to the request has been received
             * @param request that is starting
             */
            virtual void onResponseReceived(const Request& request) = 0;

            /**
             * Called when data is received from the response body
             * @param request with data
             * @param bytes number of bytes received on a network receive (non cumulative)
             */
            virtual void onResponseBytes(const Request& request, size_t bytes) = 0;

            /**
             * Called when a request response body has reached the end
             * @param request for which the response body has ended
             */
            virtual void onResponseEnd(const Request& request) = 0;

            /**
             * Called when a request errors either during the request or downloading the response
             * @param request that has errored
             * @param error error code
             */
            virtual void onRequestError(const Request& request, int error) = 0;
        };
    };

    /** Represents receiver of media sample output from a MediaSource */
    class Listener : public Request::Listener {
    public:
        ~Listener() override = default;

        /**
         * Signals the source duration has changed after the ready() event
         * @param duration new duration
         */
        virtual void onSourceDurationChanged(MediaTime duration) = 0;

        /** Signals the source has reached the end of the stream */
        virtual void onSourceEndOfStream() = 0;

        /**
         * Signals a source error (e.g. network or I/O error)
         * @param error description
         */
        virtual void onSourceError(const Error& error) = 0;

        /**
         * Signals a source error (e.g. network or I/O error) that is being recovered
         * @param error description
         */
        virtual void onSourceRecoverableError(const Error& error) = 0;

        /** Signals the end of a set of sample data */
        virtual void onSourceFlush() = 0;

        /**
         * Signals the source is opened (meaning the read() method can be called and qualities can
         * be queried)
         */
        virtual void onSourceOpened() = 0;

        /**
         * Sends a media sample to the player
         * @param track id
         * @param sample media sample
         */
        virtual void onSourceSample(int track, std::shared_ptr<MediaSampleBuffer> sample) = 0;

        /**
         * Signals a media format change, should be called once before handleSample and anytime
         * the format of the sample stream changes.
         * @param track id
         * @param format media format
         */
        virtual void onSourceTrack(int track, std::shared_ptr<MediaFormat> format) = 0;

        /**
         * Signal that the source seek capability value has changed
         * @param seekable true if the source is seekable
         */
        virtual void onSourceSeekableChanged(bool seekable) = 0;

        /**
         * Called when source needs the target quality to be set (e.g. before a segment download)
         */
        virtual void onSourceUpdateQuality() = 0;

        /**
         * Called with the Session specific info for the source
         * @param properties key-value properties for the source
         */
        virtual void onSourceSessionData(const std::map<std::string, std::string>& properties) = 0;
    };
};
}
