#pragma once

#include "playercore/MediaType.hpp"
#include <chrono>
#include <cstdint>

namespace twitch {
/** Defines a time represented as a rational number (value/scale) where scale is units in a second */
class MediaTime final {
public:
    /**
     * Creates a time at zero.
     */
    MediaTime() noexcept;

    /**
     * Creates a time with a given value and scale.
     * @param value
     * @param scale
     */
    MediaTime(int64_t value, uint32_t scale) noexcept;

    /**
     * Converts from a double seconds value
     * @param seconds value
     */
    explicit MediaTime(double seconds) noexcept;

    /**
     * Converts from std::chrono::duration
     * @param duration value
     */
    template <typename Rep, typename Period>
    MediaTime(std::chrono::duration<Rep, Period> dur)
        : MediaTime(static_cast<int64_t>(dur.count() * Period::num), Period::den)
    {
    }

    /**
     * @return The time value in timebase
     */
    int64_t count() const { return m_value; }

    uint32_t timebase() const { return m_scale; }

    /** @return zero value */
    static MediaTime zero();

    /** @return max time value*/
    static MediaTime max();

    /** @return invalid time value (same as min)*/
    static MediaTime invalid();

    /** @return MediaTime representing the current monotonic clock */
    template <typename Clock = std::chrono::high_resolution_clock>
    static MediaTime now()
    {
        auto now = Clock::now();
        return MediaTime(std::chrono::duration_cast<std::chrono::microseconds>(now.time_since_epoch()).count(), std::micro::den);
    }

    /**
     * Converts this MediaTime's value to another scale
     * @param scale desired time scale
     * @return MediaTime value scaled using the given factor
     */
    MediaTime scaleTo(uint32_t scale) const;

    /** @return MediaTime with the absolute value of this time instance. */
    MediaTime absolute() const;

    /** @return true if the time is valid (scale greater than zero or false if invalid */
    bool valid() const;

    int compare(const MediaTime& rhs) const;

    using Microseconds = std::chrono::microseconds;
    using Milliseconds = std::chrono::milliseconds;
    using Nanoseconds = std::chrono::nanoseconds;

    /** @return convert to seconds with double precision */
    double seconds() const;

    /** @return convert to std::chrono::milliseconds */
    Milliseconds milliseconds() const;

    /** @return convert to std::chrono::microseconds */
    Microseconds microseconds() const;

    /** @return convert to std::chrono::nanoseconds */
    Nanoseconds nanoseconds() const;

    bool operator==(const MediaTime& rhs) const { return compare(rhs) == 0; }
    bool operator!=(const MediaTime& rhs) const { return compare(rhs) != 0; }
    bool operator<=(const MediaTime& rhs) const { return compare(rhs) <= 0; }
    bool operator>=(const MediaTime& rhs) const { return compare(rhs) >= 0; }
    bool operator<(const MediaTime& rhs) const { return compare(rhs) < 0; }
    bool operator>(const MediaTime& rhs) const { return compare(rhs) > 0; }

    friend MediaTime operator*(MediaTime lhs, double rhs) { return lhs *= rhs; }
    friend MediaTime operator/(MediaTime lhs, double rhs) { return lhs /= rhs; }
    friend MediaTime operator+(MediaTime lhs, const MediaTime& rhs) { return lhs += rhs; }
    friend MediaTime operator-(MediaTime lhs, const MediaTime& rhs) { return lhs -= rhs; }

    MediaTime& operator*=(double rhs);
    MediaTime& operator/=(double rhs);
    MediaTime& operator+=(const MediaTime& rhs);
    MediaTime& operator-=(const MediaTime& rhs);

private:
    /** The time value */
    int64_t m_value;
    /** The time scale as represented as units in a second */
    uint32_t m_scale;
};
}
