#pragma once

#include <string>

namespace twitch {
/**
 * Represents a media type or mime type containing a top level type and a subtype e.g.
 * "video/mp4" and possible parameters. (https://tools.ietf.org/html/rfc6838)
 */
struct MediaType {
    std::string name; /** full name of the media */
    std::string type; /** top level type of the media */
    std::string subType; /** sub type of the media */
    std::string parameters; /** parameters in the media type */

    static const std::string Type_Applicaton; /** top level type application */
    static const std::string Type_Video; /** top level type video */
    static const std::string Type_Audio; /** top level type audio */
    static const std::string Type_Text; /** top level type text */

    // predefined types
    static const MediaType Application_OctetStream; /** Binary stream */
    static const MediaType Application_Apple_MPEG_URL; /** Apple HLS playlist */
    static const MediaType Application_MPEG_URL; /** HLS playlist */
    static const MediaType Application_Json; /** JSON */
    static const MediaType Binary_OctetStream; /** Binary stream */
    static const MediaType Video_AVC; /** H.264 AVC Video */
    static const MediaType Audio_AAC; /** AAC Audio */
    static const MediaType Audio_MP4; /** Audio MP4 file */
    static const MediaType Audio_PCM; /** PCM audio */
    static const MediaType Video_MP2T; /** Video MPEG2 Transport Stream */
    static const MediaType Video_MP4; /** Video MP4 file */
    static const MediaType Video_NV12; /** YUV 4:2:0 */
    static const MediaType Video_NV21; /** YUV 4:2:0 U/V reversed */
    static const MediaType Video_Quicktime; /** Quicktime Movie file */
    static const MediaType Video_VP9; /** VP9 Video */
    static const MediaType Text_VTT; /** WebVTT */
    static const MediaType Text_ID3; /** ID3 Metadata */
    static const MediaType Text_Plain; /** Plain text */
    static const MediaType Text_Json; /** JSON */

    bool operator<(const MediaType& other) const { return name < other.name; }
    bool operator>(const MediaType& other) const { return name > other.name; }
    bool operator==(const MediaType& other) const { return name == other.name; }
    bool operator!=(const MediaType& other) const { return !(*this == other); }

    MediaType() = default;
    MediaType(const std::string& mediaType);
    MediaType(const std::string& type, const std::string& subType, const std::string& parameters = std::string());

    bool isAudio() const { return type == MediaType::Type_Audio; }
    bool isVideo() const { return type == MediaType::Type_Video; }
    bool isText() const { return type == MediaType::Type_Text; }

    /**
     * Checks if one media type matches (is compatible with) another.
     * @param other the other media type
     * @return true if the types are compatible, false otherwise
     */
    bool matches(const MediaType& other) const;

    /** @return a copy of this media type without it's parameters (just type/subtype) */
    MediaType withoutParameters() const;

    /**
     * From a given path maps on to the known media types and returns the best match, generally based on the extension.
     * @param path to check
     * @return MediaType best matching the path based on the extension or an empty mediaType otherwise
     */
    static MediaType matchFromPath(const std::string& path);
};
}
