#pragma once

#include "playercore/Error.hpp"
#include "playercore/MediaSource.hpp"
#include "playercore/MediaTime.hpp"
#include "playercore/Quality.hpp"
#include "playercore/platform/Platform.hpp"

namespace twitch {
/**
 * This is the shared Player interface used across different platforms for media playback. Specific
 * platforms may extend this to provide platform specific APIs, for example setting a rendering
 * surface for displaying rendered video.
 *
 * API users instantiate a player implementation and issue commands to it. The commands are
 * generally asynchronous, the player communicates its state back to the user via the
 * Player::Listener interface. The two primary callbacks are for Player::State changes and errors.
 *
 * The following example shows using the Player to play a network stream until the source ends:
 *
 * @verbatim
 *
 *      API caller                            Player
 *          |              load(uri)            |
 *          |---------------------------------->|
 *          |                                   |
 *          |   onStateChanged(State::Ready)    |
 *          |<----------------------------------|
 *          |                                   |
 *          |               start()             |
 *          |---------------------------------->|
 *          |                                   |
 *          |   onStateChanged(State::Buffering)|
 *          |<----------------------------------|
 *          |                                   |
 *          |   onStateChanged(State::Playing)  |
 *          |<----------------------------------|
 *          |                                   |
 *          |           getPosition()           |
 *          |---------------------------------->|
 *          |                                   |
 *          |   onStateChanged(State::Ended)    |
 *          |<----------------------------------|
 *          |                                   |
 *          |               stop()              |
 *          |---------------------------------->|
 *
 * @endverbatim
 */
class Player {
public:
    virtual ~Player() = default;

    /** Player states */
    enum class State {
        /** Player is idle. This is the initial state. */
        Idle,
        /** Player is ready for playback, meaning load was successfully called. */
        Ready,
        /** Player is buffering for data from network or file. */
        Buffering,
        /** Player is playing the stream. */
        Playing,
        /** Player has reached the end of the stream. */
        Ended
    };

    /** Receives player events */
    class Listener {
    public:
        Listener() = default;
        virtual ~Listener() = default;
        /**
         * Invoked on the source duration changing. This can occur after the Ready state to indicate
         * a change in the duration of the media.
         * @param duration the current duration of the source.
         */
        virtual void onDurationChanged(MediaTime duration) = 0;

        /**
         * Invoked when there is an error in the player.
         * @param error structure describing the error
         */
        virtual void onError(const Error& error) = 0;

        /**
         * Invoked on a timed metadata event from the stream
         * @param type mime type e.g. 'text/json'
         * @param data metadata content, format depends on the type of metadata
         */
        virtual void onMetadata(const std::string& type, const std::vector<uint8_t>& data) = 0;

        /**
         * Invoked when the player's playing position has changed. The frequency at which this
         * method is called can vary per platform.
         * @param position current playing position
         */
        virtual void onPositionChanged(MediaTime position) = 0;

        /**
         * Invoked when the playing quality has changed either from a user action or from an
         * internal automatic quality switch.
         * @param quality current active quality
         */
        virtual void onQualityChanged(const Quality& quality) = 0;

        /**
         * Invoked on a buffering update where the buffer is emptied while playing excluding user
         * actions such as seeking, starting or resuming the stream.
         */
        virtual void onRebuffering() = 0;

        /**
         * Invoked when there is a recoverable error in the player.
         * @param error structure describing the error
         */
        virtual void onRecoverableError(const Error& error) = 0;

        /**
         * Invoked after the player has seeked to a given position as requested from seekTo.
         * @param time the seeked time.
         */
        virtual void onSeekCompleted(MediaTime time) = 0;

        /**
         * Invoked when session data for the current playback session, contains arbitrary
         * key-value pairs generally used for analytics and session specific data.
         * @param properties key-value properties for the source
         */
        virtual void onSessionData(const std::map<std::string, std::string>& properties) = 0;

        /**
         * Invoked on a state change in the player
         * @param state the current state
         */
        virtual void onStateChanged(State state) = 0;

        /**
         * Invoked when an analytics tracking event should be sent
         * @param name of the event
         * @param properties of the event that have been serialized from JSON
         */
        virtual void onAnalyticsEvent(const std::string& name, const std::string& properties) = 0;
    };

    /**
     * Creates a player instance.
     *
     * @param listener for player events
     * @param platform non empty instance of the platform to use
     * @return a Player instance using the given platform
     */
    static std::shared_ptr<Player> create(Player::Listener& listener, std::shared_ptr<Platform> platform);

    /** @return string representation of the player state */
    static const char* stateToString(Player::State state);

    /**
     * Loads the given stream. On success the player state will change from it's current state
     * to State::Ready. On failure invokes the onError listener method.
     *
     * @param path of the stream to load (e.g. a url or file)
     */
    virtual void load(const std::string& path) = 0;

    /**
     * Loads the given stream. On success the player state will change from it's current state to
     * State::Ready. On failure invokes the onError listener method.
     *
     * @param path of the stream to load (e.g. a url or file)
     * @param mediaType Media type of the content if known, e.g. "video/mp4" or "application/x-mpegURL"
     */
    virtual void load(const std::string& path, const std::string& mediaType) = 0;

    /**
     * Starts or resumes playback of the current stream or fails if no stream is loaded.
     * On success depending on the type of stream the player state will change to State::Buffering
     * and then State::Playing or just State::Playing. On failure invokes the onError listener
     * method.
     */
    virtual void play() = 0;

    /**
     * Pauses playback of the current stream or fails if no stream is loaded. On failure invokes the
     * onError listener method.
     *
     * Resuming the stream with play may result in the position being different depending on the
     * type of media being played.
     */
    virtual void pause() = 0;

    /**
     * Seeks to the given time in the stream and begins playing at that position if play() has been
     * called. On success depending on the type of stream the player state will change to
     * State::Buffering and then State::Playing or remain in State::Playing. getPosition() will
     * update to the seeked time. On failure invokes the onError listener method.
     *
     * @param time to seek to
     */
    virtual void seekTo(MediaTime time) = 0;

    /**
     * @return true if seeking the source is supported, false otherwise or if the player is not in
     * the State::Ready state
     */
    virtual bool isSeekable() const = 0;

    /**
     * @return total duration of the media being played back or MediaTime::max() for an unknown or
     * potentially indefinite length stream.
     */
    virtual MediaTime getDuration() const = 0;

    /** @return current playing position */
    virtual MediaTime getPosition() const = 0;

    /** @return current buffered position */
    virtual MediaTime getBufferedPosition() const = 0;

    /** @return current state of the player */
    virtual State getState() const = 0;

    /** @return Playback statistics for the stream currently playing */
    virtual const Statistics& getStatistics() const = 0;

    /** @return true if automatic quality switching is enabled, false otherwise */
    virtual bool getAutoSwitchQuality() const = 0;

    /**
     * Enables automatic quality selection (based on the device/network) when possible
     * @param enable true to enable, false to disable
     */
    virtual void setAutoSwitchQuality(bool enable) = 0;

    /** @return the current quality of the source */
    virtual const Quality& getQuality() const = 0;

    /**
     * Sets the current quality of the stream, if auto switch quality it is disabled
     * @param quality a valid quality entry from getQualities()
     */
    virtual void setQuality(const Quality& quality) = 0;

    /**
     * Sets the current quality of the stream, if auto switch quality it is disabled.
     * @param quality a valid quality entry from getQualities()
     * @param bool adaptive, true for an adaptive quality change (quality changes at the end
     * of the buffer, false otherwise buffer cleared immediately)
     */
    virtual void setQuality(const Quality& quality, bool adaptive) = 0;

    /** @return the available qualities or empty if none are currently available */
    virtual const std::vector<Quality>& getQualities() const = 0;

    /** @return average source media bitrate of the stream */
    virtual int getAverageBitrate() const = 0;

    /** @return for network streams the current ABR bandwidth estimate in bps or 0 if unknown */
    virtual int getBandwidthEstimate() const = 0;

    /** @return current playback rate defaults to 1.0 */
    virtual float getPlaybackRate() const = 0;

    /**
     * Sets the current playback rate of the stream.
     * @param rate playback rate to set
     */
    virtual void setPlaybackRate(float rate) = 0;

    /**
     * @return an identifier for the type of player being used in the case of multiple Player
     * implementations, the default implementation value is "mediaplayer"
     */
    virtual const std::string& getName() const = 0;

    /** @return source path or uri currently loaded or being played, empty if none */
    virtual std::string getPath() const = 0;

    /** @return version string of the player */
    virtual const std::string& getVersion() const = 0;

    /**
     * Sets the initial starting bitrate when using auto quality switching. This determines the
     * initial quality an adaptive stream will playback at when no bandwidth estimate value has
     * been determined.
     * @param bitrate initial bitrate to use
     */
    virtual void setAutoInitialBitrate(int bitrate) = 0;

    /**
     * Sets the max bitrate to use when using auto quality switching. Allows controlling resource
     * usage in the case of multiple players.
     * @param bitrate initial bitrate to use
     */
    virtual void setAutoMaxBitrate(int bitrate) = 0;

    /**
     * Sets the max video display size of the player, which prevents switching to qualities above
     * this resolution when autoSwitchQuality is true.
     * @param width display width
     * @param height display height
     */
    virtual void setAutoMaxVideoSize(int width, int height) = 0;

    /**
     * Sets a target live latency, during buffering events if the latency would exceed this value
     * the live stream will be re-syncd to the live window to prevent the latency exceeding this
     * value.
     * @param time max latency before attempting to skip content during live playback
     */
    virtual void setLiveMaxLatency(MediaTime time) = 0;

    /**
     * Enables low latency playback for streams which support it. Defaults to true, changing the
     * value during playback will restart the stream.
     *
     * @param enabled true to enable, false to disable
     */
    virtual void setLiveLowLatencyEnabled(bool enabled) = 0;

    /**
     * Sets the minimum buffer size needed to start playback.
     * @param duration of buffer to set
     */
    virtual void setMinBuffer(MediaTime duration) = 0;

    /**
     * Sets the target buffer size to buffer ahead of the playhead. Note in live streams this
     * buffer size may not be met.
     * @param duration of buffer to set
     */
    virtual void setMaxBuffer(MediaTime duration) = 0;

    /**
     * Sets the video surface to render video to. The type of pointer is platform dependent.
     * @param surface handle to a surface object to set.
     */
    virtual void setSurface(void* surface) = 0;

    /** @return For live streams returns the latency to the source stream if known */
    virtual MediaTime getLiveLatency() const = 0;

    /** @return true if setLooping has been set to true, false otherwise */
    virtual bool isLooping() const = 0;

    /**
     * Enables the player to loop the source video if supported
     * @param loop true to loop video playback, false otherwise
     */
    virtual void setLooping(bool loop) = 0;

    /** @return true if the player is muted, false otherwise */
    virtual bool isMuted() const = 0;

    /**
     * Mutes or unmutes the player
     * @param muted true to mute the stream false to unmute
     */
    virtual void setMuted(bool muted) = 0;

    /** @return the current volume set for the player */
    virtual float getVolume() const = 0;

    /**
     * Sets the playback volume of the audio track for the player (either for the current stream or
     * the next stream that is loaded).
     * @param volume value to set between 0.0f and 1.0f
     */
    virtual void setVolume(float volume) = 0;

    /**
     * Sets the API client id used for internal requests made by the player
     * @param id to set
     */
    virtual void setClientId(const std::string& id) = 0;

    /**
     * Sets the device id for this player instance used for API requests
     * @param id to set
     */
    virtual void setDeviceId(const std::string& id) = 0;

    /**
     * Sets the Twitch auth token used for internal API requests
     * @param token to set
     */
    virtual void setAuthToken(const std::string& token) = 0;

    /**
     * Sets the type of the player as a string.
     * @param type to set
     */
    virtual void setPlayerType(const std::string& type) = 0;

    /**
     * Requests a server ad roll.
     */
    virtual void requestServerAd() = 0;

    /**
     * Explicitly triggers a new play session, resetting tracking state.
     */
    virtual void regenerateAnalyticsPlaySession() = 0;

    /**
     * Sets whether or not calls to load() will automatically start a new play session.
     */
    virtual void setKeepAnalyticsPlaySession(bool keepPlaySession) = 0;
};
}
