#pragma once

#include "playercore/MediaTime.hpp"

namespace twitch {
/**
 * Represents a range of time using a start MediaTime and a MediaTime duration.
 */
struct TimeRange {
    /** start time of the time range */
    MediaTime start = MediaTime::zero();

    /** duration of the time range */
    MediaTime duration = MediaTime::zero();

    /**
     * Creates a empty time range with a zero start time and a zero duration.
     */
    TimeRange() = default;

    /**
     * Creates a new time range with a given start time and duration.
     *
     * @param start the start time in absolute time
     * @param duration the duration of the range
     */
    TimeRange(MediaTime start, MediaTime duration)
        : start(start)
        , duration(duration)
    {
    }

    /** @return The end time of the current range in absolute time based on the start time value */
    MediaTime end() const { return start + duration; }

    /**
     * Checks if a time is inside the range
     * @param time to check
     * @return true if this time falls inside the range, that is greater than or equal than the
     * start time and less then the end time
     */
    bool contains(MediaTime time) const { return time >= start && time < end(); }

    bool operator==(const TimeRange& other) const { return start == other.start && duration == other.duration; }
    bool operator!=(const TimeRange& other) const { return !(*this == other); }
    bool operator<(const TimeRange& other) const { return other.start >= end(); }
    bool operator>(const TimeRange& other) const { return start >= other.end(); }
};
}
