#pragma once

#include "playercore/platform/Platform.hpp"
#include <cstdint>
#include <string>
#include <vector>

namespace twitch {
/**
 * Common interface for requesting access to protected media content via a platform decryption
 * module. When the player encounters protected media an implementation of this interface is
 * created for the given protection scheme.
 *
 * The implementation is responsible for generating opaque license key and device certificate
 * requests. The player will execute these requests and provide the response data back to the
 * implementation. The implementation signals key expiration/requests via a callback interface.
 *
 * Examples of platform modules:
 * Web EME
 * https://www.w3.org/TR/encrypted-media/
 * Android MediaDrm
 * https://developer.android.com/reference/android/media/MediaDrm.html
 * PlayReady MediaProtectionManager
 * https://docs.microsoft.com/en-us/uwp/api/Windows.Media.Protection.MediaProtectionManager
 * See ISO/IEC 23001-7: Common Encryption standard
 */
class DrmSession {
public:
    virtual ~DrmSession() = default;

    /**
     * Listener for events from the session, including changes to the key status (expiration,
     * renewal, errors).
     */
    class Listener {
    public:
        virtual ~Listener() = default;
        /**
         * Callback indicating the DRM module current key status has changed
         * @param session containing the key change
         */
        virtual void onKeyExpired(DrmSession& session) = 0;

        /**
         * Callback indicating the DRM module needs a key request to be executed.
         * @param session requesting keys
         */
        virtual void onKeyRequested(DrmSession& session) = 0;

        /**
         * Callback indicating the DRM module needs a provision request to be executed.
         * @param session requesting provisioning
         */
        virtual void onProvisioningRequested(DrmSession& session) = 0;

        /**
         * Callback indicating an action on the DRM session has failed.
         * @param error indicating the failure
         */
        virtual void onError(DrmSession& session, const Error& error) = 0;
    };

    /**
     * Initializes the session, at which point any required provisioning steps should be made.
     */
    virtual void initialize() = 0;

    // https://www.w3.org/TR/encrypted-media/#dom-mediakeysession-sessionid
    // https://developer.android.com/reference/android/media/MediaDrm.html#openSession()
    /**
     * Gets the session id for this instance. The session id is used to correlate encrypted media
     * samples sent to a secure decoder instance with a specific key session.
     *
     * @return session id for this instance
     */
    virtual const std::string& getSessionId() const = 0;

    /**
     * Contains the url and the opaque data the player sends to a license server either for a key
     * or certificate.
     */
    struct OpaqueRequest {
        std::string url;
        std::vector<uint8_t> data;
    };

    // https://www.w3.org/TR/encrypted-media/#dom-mediakeysession-generaterequest
    // https://developer.android.com/reference/android/media/MediaDrm.KeyRequest.html
    /**
     * Generates a license key request.
     *
     * @param init initialization data
     * @return request to execute
     */
    virtual OpaqueRequest generateKeyRequest(const std::vector<uint8_t>& init) = 0;

    // https://developer.android.com/reference/android/media/MediaDrm.ProvisionRequest.html
    /**
     * Generates a device certificate provisioning request.
     *
     * @return request to execute
     */
    virtual OpaqueRequest generateProvisionRequest() = 0;

    // https://www.w3.org/TR/encrypted-media/#dom-mediakeysession-update
    // https://developer.android.com/reference/android/media/MediaDrm.html#provideKeyResponse(byte[], byte[])
    /**
     * Provides the license key request response.
     *
     * @param data response data
     */
    virtual void updateKeyResponse(const std::vector<uint8_t>& data) = 0;

    /**
     * Provides the device provision request response.
     *
     * @param data response data
     */
    virtual void updateProvisionResponse(const std::vector<uint8_t>& data) = 0;
};
}
