#pragma once

#include <algorithm>
#include <cctype>
#include <chrono>
#include <functional>
#include <map>
#include <memory>
#include <string>
#include <vector>

#include "playercore/platform/HttpRequest.hpp"
#include "playercore/platform/HttpResponse.hpp"

namespace twitch {
/** Represents an HTTP client which can execute HttpRequest instances and return a HttpResponse. */
class HttpClient {
public:
    virtual ~HttpClient() = default;

    /**
     *  Creates a HTTP request instance.
     *  @param url request url
     *  @param method request method to use
     */
    virtual std::shared_ptr<HttpRequest> createRequest(const std::string& url, HttpMethod method) = 0;

    /** Error callback function if sending the request fails. */
    using ErrorHandler = std::function<void(int error)>;

    /** Response callback called after reading the response code and headers */
    using ResponseHandler = std::function<void(std::shared_ptr<HttpResponse> response)>;

    /**
     *  Sends an HttpRequest asynchronously or synchronously invoking the callback methods on
     *  success or an error.
     *
     *  @param request to send
     *  @param onResponse response handler
     *  @param onError error handler
     */
    virtual void send(std::shared_ptr<HttpRequest> request, ResponseHandler onResponse, ErrorHandler onError) = 0;
};

struct CaseInsensitiveStringComparator {
    bool operator()(const std::string& a, const std::string& b) const
    {
        return std::lexicographical_compare(a.begin(), a.end(), b.begin(), b.end(), [](unsigned char lhs, unsigned char rhs) {
            return std::tolower(lhs) < std::tolower(rhs);
        });
    }
};
using HttpHeaderMap = std::map<std::string, std::string, CaseInsensitiveStringComparator>;
}
