#pragma once

#include <chrono>
#include <functional>
#include <memory>
#include <string>
#include <vector>

#include "playercore/platform/HttpMethod.hpp"

namespace twitch {
/** Represents an HTTP request that can be sent through HttpClient::send. */
class HttpRequest {
public:
    virtual ~HttpRequest() = default;

    /** @return destination url of the request. */
    const std::string& getUrl() const { return m_url; }

    /** @return request method string */
    const std::string& getMethodString() const { return m_methodString; }

    /**
     * Sets a given request header.
     * @param key header key
     * @param value header value
     */
    virtual void setHeader(const std::string& key, const std::string& value) = 0;

    /**
     * Sets a request mode.
     * @param key header key
     * @param value header value
     */
    void setRequestMode(const std::string& mode) { m_requestMode = mode; };

    /**
     *  sets the content for the request
     *  @param content of the request
     */
    virtual void setContent(const std::vector<uint8_t>& content) = 0;

    /** gets the timeout for the request */
    std::chrono::seconds getTimeout() { return m_timeout; }

    /**
     *  sets the timeout for the request
     *  @param timeout to set
     */
    void setTimeout(std::chrono::seconds timeout) { m_timeout = timeout; }

    /** Cancels a request that is in progress */
    virtual void cancel() = 0;

protected:
    /**
     *  Creates a HTTP request instance.
     *  @param url request url
     *  @param method request method to use
     */
    explicit HttpRequest(const std::string& url, HttpMethod method)
        : m_url(url)
        , m_method(method)
        , m_methodString(httpMethodToString(method))
        , m_requestMode("")
        , m_timeout(std::chrono::seconds(10))
    {
    }

    std::string m_url;
    HttpMethod m_method;
    std::string m_methodString;
    std::string m_requestMode;
    std::chrono::seconds m_timeout;
};
}
