#pragma once

#include <chrono>
#include <functional>
#include <map>
#include <memory>
#include <string>
#include <vector>

#include "playercore/platform/HttpStatus.hpp"

namespace twitch {
/** Represents an HTTP response returned from HttpClient::send. */
class HttpResponse {
public:
    virtual ~HttpResponse() = default;

    /**
     * Returns the header value for the given name, or empty string if not present.
     * @param key header key
     * @return headers received in the response message.
     */
    virtual std::string getHeader(const std::string& key) const = 0;

    /** @return HTTP status code return in the request. */
    int getStatus() const { return m_status; }

    /** @return true if the status code is successful (status is 2XX) */
    bool isSuccess() const { return getStatus() >= HttpStatus::HttpStatusOk && getStatus() < HttpStatus::HttpStatusMultipleChoices; }

    /** Gets the read timeout  */
    std::chrono::seconds getReadTimeout() const { return m_timeout; }

    /** Response content handling function receives byte data as it is read from the response. */
    using ContentHandler = std::function<void(const uint8_t* buffer, size_t size, bool endOfStream)>;

    /** Error callback function if reading from the response fails. */
    using ErrorHandler = std::function<void(int error)>;

    /**
     *  Sets the read timeout for reading the body of the response
     *  @param timeout to set
     */
    virtual void setReadTimeout(std::chrono::seconds timeout) { m_timeout = timeout; }

    /**
     *  Reads the content stream from the response until it is completed or an error occurs.
     *  @param onBuffer data buffer callback function
     *  @param onError optional error callback function
     */
    virtual void read(ContentHandler onBuffer, ErrorHandler onError) = 0;

protected:
    void setStatus(int status) { m_status = status; }
    explicit HttpResponse(int status)
        : m_status(status)
        , m_timeout(std::chrono::seconds(10))
    {
    }

private:
    int m_status;
    std::chrono::seconds m_timeout;
};
}
