#pragma once

#include "playercore/Log.hpp"
#include "playercore/MediaDecoder.hpp"
#include "playercore/MediaRenderer.hpp"
#include "playercore/ReferenceClock.hpp"
#include "playercore/platform/DrmSession.hpp"
#include "playercore/platform/Platform.hpp"
#include <functional>
#include <thread>

namespace twitch {
/** Native platform interface */
class NativePlatform : public Platform {
public:
    NativePlatform() = default;
    ~NativePlatform() override = default;
    NativePlatform& operator=(const NativePlatform&) = delete;
    NativePlatform(const NativePlatform&) = delete;

    /**
     * Creates a non-blocking HttpClient on a separate thread that executes callbacks on the give scheduler
     * @param scheduler on which the data and error callbacks should be executed on
     * @return a HttpClient instance
     */
    std::shared_ptr<HttpClient> createAsyncHttpClient(std::shared_ptr<Scheduler> scheduler) override;

    /** @return a new Scheduler instance executing tasks on a single thread. */
    std::shared_ptr<Scheduler> createScheduler(const std::string& name) override;

    /**
     * Creates a a media sink instance for the player, defaults to decode-render pipeline per track.
     * @param listener sink listener
     * @param scheduler to use for async callbacks
     * @return MediaSink instance.
     */
    std::unique_ptr<MediaSink>
    createSink(MediaSink::Listener& listener, std::shared_ptr<Scheduler> scheduler) override;

    /**
     * Creates a a media source instance for the player.
     * @param path url or file path to the media
     * @param listener for the source
     * @param scheduler to use for async callbacks
     * @return MediaSource instance for the given path or nullptr if not supported.
     */
    std::unique_ptr<MediaSource>
    createSource(const std::string& path, const MediaType& mediaType, MediaSource::Listener& listener, std::shared_ptr<Scheduler> scheduler) override;

    /**
     * Creates a decoder instance for the player.
     * @param format of the media to decode
     * @return Decoder instance for the given format instance or nullptr if not supported.
     */
    virtual std::unique_ptr<MediaDecoder> createDecoder(std::shared_ptr<const MediaFormat> format) = 0;

    /**
     * Creates a renderer instance for the player.
     * @param clock reference clock for time synchronization
     * @param format of the media to render
     * @return Renderer instance for the given format instance or nullptr if not supported.
     */
    virtual std::unique_ptr<MediaRenderer> createRenderer(const ReferenceClock& clock, std::shared_ptr<const MediaFormat> format) = 0;

    /**
     * Creates a DRM session object for the given key system. If the platform doesn't support
     * the given key system it should return an empty pointer.
     * @param system UUID
     * @param listener session listener callback
     * @return session to use or null if the system isn't supported
     */
    virtual std::unique_ptr<DrmSession> createDrmSession(const std::vector<uint8_t>& system, DrmSession::Listener& listener);

    /** @return the set of media types supported for rendering by the platform */
    const std::set<MediaType>& getSupportedMediaTypes() const override;

    /** @return temporary directory path for storing temporary files used by the player */
    virtual const std::string& getTempPath() const = 0;

    const std::set<std::vector<uint8_t>>& getSupportedProtectionSystems() const override;

    const std::map<std::string, std::string>& getAnalyticsProperties() override;

    std::shared_ptr<MediaReaderFactory> getMediaReaderFactory() const override;

    VideoDecoderCapabilities getVideoDecoderCapabilities(const MediaType& mediaType) override;

    /** Handler function that will be called every time a new thread is created within the PlayerCore library */
    virtual void onThreadCreated(std::thread::id id, const std::string& name);

    /**
     * Set the name of the current thread
     * @param name to set
     */
    virtual void setCurrentThreadName(const std::string& name) = 0;

    /** Sets the log message handler that override default logging message */
    using LogHandler = std::function<void(Log::Level level, const std::string& message)>;
    static void setLogMessageHandler(LogHandler onLog);

    /** Gets the log message handler */
    static LogHandler getLogMessageHandler();
};
}
