#pragma once

#include "playercore/Log.hpp"
#include "playercore/MediaSink.hpp"
#include "playercore/MediaSource.hpp"
#include "playercore/MediaType.hpp"
#include "playercore/platform/Capabilities.hpp"
#include "playercore/platform/HttpClient.hpp"
#include "playercore/platform/Scheduler.hpp"
#include "playercore/platform/VideoDecoderCapabilities.hpp"
#include <limits>
#include <set>
#include <vector>

namespace twitch {
class MediaReaderFactory;
/**
 * Platform abstraction interface. Provides access to platform specific APIs and features through a
 * common set of interfaces. This includes networking, logging, event scheduling as well as the
 * capabilities of the platform in terms of media formats, codecs and protection systems.
 */
class Platform {
public:
    virtual ~Platform() = default;

    /**
     * Creates a non-blocking HttpClient that can execute callbacks on a particular scheduler
     * @param scheduler on which the data and error callbacks should be executed on
     * @return a HttpClient instance
     */
    virtual std::shared_ptr<HttpClient> createAsyncHttpClient(std::shared_ptr<Scheduler> scheduler) = 0;

    /** @return a new Scheduler instance. */
    virtual std::shared_ptr<Scheduler> createScheduler(const std::string& name) = 0;

    /**
     * Creates a media sink instance for the player.
     * @param listener for the sink
     * @param scheduler to use for async callbacks
     * @return MediaSink instance for the given format.
     */
    virtual std::unique_ptr<MediaSink>
    createSink(MediaSink::Listener& listener, std::shared_ptr<Scheduler> scheduler) = 0;

    /**
     * Creates a media source instance for the player.
     * @param path url or file path to the media
     * @param mediaType media type of the content if known
     * @param listener for the source
     * @param scheduler to use for async callbacks
     * @return MediaSource instance for the given path or nullptr if not supported.
     */
    virtual std::unique_ptr<MediaSource>
    createSource(const std::string& path, const MediaType& mediaType, MediaSource::Listener& listener, std::shared_ptr<Scheduler> scheduler) = 0;

    /** @return a Log instance which can output to the platform log. */
    virtual std::shared_ptr<Log> getLog() const = 0;

    /** @return a HttpClient instance which can be used to create and send HttpRequest instances. */
    virtual std::shared_ptr<HttpClient> getHttpClient() const = 0;

    /** @return a string instance which is the name of the platform. */
    virtual const std::string& getName() const = 0;

    /**
     * Given a video format media type determines the supported capabilities for that type.
     *
     * @param mediaType media type of the video, this is guaranteed to be a video type and
     *        only one of the types returned from getSupportedMediaTypes()
     * @return capabilities out parameter of the VideoDecodeCapabilities for the given type.
     */
    virtual VideoDecoderCapabilities getVideoDecoderCapabilities(const MediaType& mediaType) = 0;

    /** @return device playback capabilities, used to limit what stream types are playable */
    virtual const Capabilities& getCapabilities() const = 0;

    /** @return the set of media types supported for rendering by the platform */
    virtual const std::set<MediaType>& getSupportedMediaTypes() const = 0;

    /** @return the set of protection scheme UUIDs supported by the platform */
    virtual const std::set<std::vector<uint8_t>>& getSupportedProtectionSystems() const = 0;

    /** @return analytic properties specific to the platform */
    virtual const std::map<std::string, std::string>& getAnalyticsProperties() = 0;

    virtual std::shared_ptr<MediaReaderFactory> getMediaReaderFactory() const = 0;
};
}
