#pragma once

#include "playercore/Cancellable.hpp"
#include <chrono>
#include <functional>
#include <memory>

namespace twitch {
/**
 * Scheduler provides an interface for both the asynchronous and synchronous execution of a code
 * block function. This can be used to schedule a function to be invoked at future time through a
 * delay or to ensure thread safety by running blocks only one thread.
 */
class Scheduler {
public:
    virtual ~Scheduler() = default;

    using Action = std::function<void()>;
    using Microseconds = std::chrono::microseconds;

    /**
     * Runs the action function in the scheduler returning a Cancellable shared instance which can
     * be used to cancel the given task.
     *
     * @param action to run
     * @return a cancellable reference to the operation
     */
    std::shared_ptr<Cancellable> schedule(Action action)
    {
        return schedule(std::move(action), Microseconds::zero(), false);
    }

    /**
     * Runs the action function in the scheduler returning a Cancellable shared instance which can
     * be used to cancel the given task.
     *
     * @param action to run
     * @param delay time delay before the given action should be run
     * @return a cancellable reference to the operation
     */
    std::shared_ptr<Cancellable> schedule(Action action, Microseconds delay)
    {
        return schedule(std::move(action), delay, false);
    }

    /**
     * Runs the action function in the scheduler returning a Cancellable shared instance which can
     * be used to cancel the given task.
     *
     * @param action to run
     * @param delay time delay before the given action should be run
     * @param repeating true if the task is repeating
     * @return a cancellable reference to the operation
     */
    virtual std::shared_ptr<Cancellable> schedule(Action action, Microseconds delay, bool repeating) = 0;

    /**
     * Runs the action function in the scheduler blocks the current thread until the action is
     * completed.
     *
     * @param action to run
     */
    virtual void scheduleAndWait(Action action) = 0;
};
}
