#pragma once
#include "Configuration.hpp"
#include "playercore/Player.hpp"
#include <memory>

namespace twitch {
class VideoFrame;
class VideoFrameBuffer;
class PS4Platform;
namespace ps4 {
class TempDirMount;
class PlayerCore {
public:
    /**
     * Callback for when the VideoFrameBuffer is available
     */
    using OnVideoFrameBuffer = std::function<void(const std::shared_ptr<twitch::VideoFrameBuffer>&)>;

    /**
     * Texture memory is allocated by PlayerCore and shared with the app for faster display.
     * PlayerCore may wants to deallocate all this memory suddenly if it is being destroyed
     * or if an error occurs in the VideoDecoder that requires reallocation.
     * When this callback is called, the app is responsible to give up all shared_ptr instances it holds on the
     * videoFrame
     * @param videoFrame the frame that is being freed.
     */
    using OnFrameNeedsReleasing = std::function<void(const std::shared_ptr<twitch::VideoFrame>& videoFrame)>;

    /**
     * Initialize the Player Core library.  Initialization should only be done once
     * @param configuration options
     * @return true if the Player Core library initialization was successful, otherwise false
     */
    static bool initialize(Configuration configuration = Configuration());

    /**
     * Checks if the Player Core library has been initialized or not
     * @return true if the Player Core library has already initialized, otherwise false
     */
    static bool isInitialized();

    /**
     * Create a Player instance for video playback
     * @param listener player event listeners
     * @param onVideoFrameBuffer callback function for passing back the VideoFrameBuffer
     * @return twitch::Player instance if the player was successfully created, otherwise nullptr
     */
    static std::shared_ptr<twitch::Player> create(twitch::Player::Listener& listener, OnVideoFrameBuffer onVideoFrameBuffer, OnFrameNeedsReleasing onFrameNeedsReleasing,
        MediaSource::Request::Listener* requestListener = nullptr);

private:
    static std::unique_ptr<ps4::Configuration> m_configuration;
    static std::unique_ptr<ps4::TempDirMount> m_tempDirMount;

    // Player Core internal
    friend class twitch::PS4Platform;
    static const std::string& getTempPath();
};
}
}
