#include "MediaRendererJNI.hpp"
#include "MediaDecoderJNI.hpp"
#include "package.hpp"

namespace twitch {
jmethodID MediaRendererJNI::s_configure;
jmethodID MediaRendererJNI::s_flush;
jmethodID MediaRendererJNI::s_release;
jmethodID MediaRendererJNI::s_render;
jmethodID MediaRendererJNI::s_start;
jmethodID MediaRendererJNI::s_stop;
jmethodID MediaRendererJNI::s_getDroppedFrames;
jmethodID MediaRendererJNI::s_getRenderedFrames;
jmethodID MediaRendererJNI::s_getRenderedTime;
jmethodID MediaRendererJNI::s_setMediaTime;
jmethodID MediaRendererJNI::s_setSurface;
jmethodID MediaRendererJNI::s_setVolume;
jmethodID MediaRendererJNI::s_setPlaybackRate;
jmethodID MediaRendererJNI::s_handleException;

void MediaRendererJNI::initialize(JNIEnv* env)
{
    jclass cls = FindPlayerClass(env, "MediaRenderer");
    s_release = env->GetMethodID(cls, "release", "()V");
    s_render = env->GetMethodID(cls, "render", "(Ljava/nio/ByteBuffer;IJ)V");
    s_configure = env->GetMethodID(cls, "configure", "(Landroid/media/MediaFormat;)V");
    s_flush = env->GetMethodID(cls, "flush", "()V");
    s_start = env->GetMethodID(cls, "start", "()V");
    s_stop = env->GetMethodID(cls, "stop", "()V");
    s_getRenderedTime = env->GetMethodID(cls, "getRenderedPresentationTime", "()J");
    s_setPlaybackRate = env->GetMethodID(cls, "setPlaybackRate", "(F)V");
    cls = FindPlayerClass(env, "AudioRenderer");
    s_setVolume = env->GetMethodID(cls, "setVolume", "(F)V");
    cls = FindPlayerClass(env, "VideoRenderer");
    s_getDroppedFrames = env->GetMethodID(cls, "getDroppedFrames", "()I");
    s_getRenderedFrames = env->GetMethodID(cls, "getRenderedFrames", "()I");
    s_setSurface = env->GetMethodID(cls, "setSurface", "(Landroid/view/Surface;)V");
    s_setMediaTime = env->GetMethodID(cls, "setMediaTime", "(J)V");

    jclass platformClass = FindPlayerClass(env, "Platform");
    s_handleException = env->GetMethodID(platformClass, "handleRendererException", "(Ljava/lang/Throwable;)V");
}

MediaRendererJNI::MediaRendererJNI(JNIEnv* env, jobject platform, jobject renderer)
    : m_env(env)
    , m_platform(env, platform)
    , m_object(env, renderer)
{
}

MediaRendererJNI::~MediaRendererJNI()
{
    if (m_object) {
        m_env->CallVoidMethod(m_object, s_release);
        checkException(); // clear any release exception
    }
}

MediaResult MediaRendererJNI::checkException(MediaResult result)
{
    if (m_env->ExceptionCheck()) {
        jthrowable throwable = m_env->ExceptionOccurred();
        m_env->ExceptionClear();
        m_env->CallVoidMethod(m_platform.get(), s_handleException, throwable);
        return MediaResult::Error;
    }

    return result;
}

MediaResult MediaRendererJNI::configure(const MediaFormat& format)
{
    if (!m_object || !m_env) {
        return MediaResult::ErrorInvalidState;
    }

    jobject javaFormat = MediaDecoderJNI::createMediaFormat(m_env, format);
    m_env->CallVoidMethod(m_object, s_configure, javaFormat);
    m_env->DeleteLocalRef(javaFormat);
    return checkException();
}

MediaResult MediaRendererJNI::flush()
{
    if (!m_object || !m_env) {
        return MediaResult::ErrorInvalidState;
    }

    m_env->CallVoidMethod(m_object, s_flush);
    return checkException();
}

MediaResult MediaRendererJNI::render(std::shared_ptr<const MediaSample> input)
{
    if (!m_object || !m_env) {
        return MediaResult::ErrorInvalidState;
    }

    if (input->type == MediaSample::Type::MemoryBuffer) {
        auto sampleBuffer = std::static_pointer_cast<const MediaSampleBuffer>(input);
        if (sampleBuffer->buffer.empty()) {
            return MediaResult::Ok;
        }

        jlong timeUs = input->decodeTime.microseconds().count();
        jobject byteBuffer = nullptr;

        uint8_t* data = const_cast<uint8_t*>(sampleBuffer->buffer.data());
        int size = static_cast<int>(sampleBuffer->buffer.size());

        if (size && data) {
            byteBuffer = m_env->NewDirectByteBuffer(data, size);
        }

        m_env->CallVoidMethod(m_object, s_render, byteBuffer, size, timeUs);
        m_env->DeleteLocalRef(byteBuffer);
        return checkException();
    } else {
        return MediaResult::Ok;
    }
}

MediaResult MediaRendererJNI::start()
{
    if (!m_object || !m_env) {
        return MediaResult::ErrorInvalidState;
    }

    m_env->CallVoidMethod(m_object, s_start);
    return checkException();
}

MediaResult MediaRendererJNI::stop()
{
    if (!m_object || !m_env) {
        return MediaResult::ErrorInvalidState;
    }

    m_env->CallVoidMethod(m_object, s_stop);
    return checkException();
}

MediaResult MediaRendererJNI::getRenderedPresentationTime(MediaTime& time)
{
    if (!m_object || !m_env) {
        return MediaResult::ErrorInvalidState;
    }

    jlong timeUs = m_env->CallLongMethod(m_object, s_getRenderedTime);

    if (timeUs == -1) {
        time = MediaTime::invalid();
    } else {
        time = MediaTime(timeUs, std::micro::den);
    }
    return checkException();
}

MediaResult MediaRendererJNI::setPlaybackRate(float rate)
{
    if (!m_object || !m_env) {
        return MediaResult::ErrorInvalidState;
    }

    m_env->CallVoidMethod(m_object, s_setPlaybackRate, static_cast<jfloat>(rate));
    return checkException();
}

}
