#include "PlatformJNI.hpp"
#include "ATrace.hpp"
#include "AttachThread.hpp"
#include "AudioRendererJNI.hpp"
#include "DrmSessionJNI.hpp"
#include "LogCat.hpp"
#include "MediaDecoderJNI.hpp"
#include "MediaRendererJNI.hpp"
#include "VideoRendererJNI.hpp"
#include "http_jni.hpp"
#include "package.hpp"
#include <jni.h>
#include <pthread.h>

namespace twitch {
jmethodID PlatformJNI::s_createDecoder;
jmethodID PlatformJNI::s_createRenderer;
jmethodID PlatformJNI::s_getProtectionSystemUUIDs;
jmethodID PlatformJNI::s_getVideoDecoderCapabilities;
jmethodID PlatformJNI::s_onThreadCreated;

jfieldID PlatformJNI::s_capabilitiesMaxBitrate;
jfieldID PlatformJNI::s_capabilitiesMaxFramerate;
jfieldID PlatformJNI::s_capabilitiesMaxWidth;
jfieldID PlatformJNI::s_capabilitiesMaxHeight;
jfieldID PlatformJNI::s_capabilitiesMaxProfile;
jfieldID PlatformJNI::s_capabilitiesMaxLevel;
jmethodID PlatformJNI::s_capabilitiesVP9Supported;

void AndroidPlatform::initialize(JavaVM* vm)
{
    PlatformJNI::initialize(vm);
}

std::shared_ptr<AndroidPlatform> AndroidPlatform::create(JNIEnv* env, jobject context)
{
    return std::make_shared<PlatformJNI>(env, context, FindPlayerClass(env, "Platform"));
}

void PlatformJNI::initialize(JavaVM* vm)
{
    jni::setVM(vm);
    JNIEnv* env = 0;
    vm->GetEnv((void**)&env, JNI_VERSION_1_6);

    jclass cls = FindPlayerClass(env, "Platform");
    std::string createDecoder = "(Landroid/media/MediaFormat;)L" + PlayerPackage + "MediaDecoder;";
    std::string createRenderer = "(Landroid/media/MediaFormat;)L" + PlayerPackage + "MediaRenderer;";
    std::string getDecoderCapabilities = "(Ljava/lang/String;)L" + PlayerPackage + "VideoCapabilities;";
    std::string getProtectionSchemeUUIDs = "()[Ljava/nio/ByteBuffer;";
    s_createDecoder = env->GetMethodID(cls, "createDecoder", createDecoder.c_str());
    s_createRenderer = env->GetMethodID(cls, "createRenderer", createRenderer.c_str());
    s_getVideoDecoderCapabilities = env->GetMethodID(cls, "getVideoDecoderCapabilities", getDecoderCapabilities.c_str());
    s_getProtectionSystemUUIDs = env->GetStaticMethodID(cls, "getSupportedProtectionSystemUUIDs", getProtectionSchemeUUIDs.c_str());
    s_onThreadCreated = env->GetMethodID(cls, "onThreadCreated", "(Ljava/lang/String;)V");

    jclass capabilitiesClass = FindPlayerClass(env, "VideoCapabilities");
    s_capabilitiesMaxBitrate = env->GetFieldID(capabilitiesClass, "maxBitrate", "I");
    s_capabilitiesMaxFramerate = env->GetFieldID(capabilitiesClass, "maxFramerate", "I");
    s_capabilitiesMaxWidth = env->GetFieldID(capabilitiesClass, "maxWidth", "I");
    s_capabilitiesMaxHeight = env->GetFieldID(capabilitiesClass, "maxHeight", "I");
    s_capabilitiesMaxProfile = env->GetFieldID(capabilitiesClass, "maxProfile", "I");
    s_capabilitiesMaxLevel = env->GetFieldID(capabilitiesClass, "maxLevel", "I");
    s_capabilitiesVP9Supported = env->GetStaticMethodID(capabilitiesClass, "isVP9Supported", "()Z");

    jni::AttachThread::initialize();
    ATrace::initialize();
    HttpClientJNI::initialize(env);
    MediaDecoderJNI::initialize(env);
    MediaRendererJNI::initialize(env);

    // DRM only supported on JellyBean MR2 and later
    if (getApiLevel() >= __ANDROID_API_J_MR2__) {
        DrmSessionJNI::initialize(env);
    }
}

PlatformJNI::PlatformJNI(JNIEnv* env, jobject context, jclass platformClass)
    : PlatformJNI(env, env->NewObject(platformClass, env->GetMethodID(platformClass, "<init>", "(Landroid/content/Context;)V"), context))
{
}

PlatformJNI::PlatformJNI(JNIEnv* env, jobject platform)
    : m_name("android")
    , m_log(std::make_shared<Logcat>())
    , m_httpClient(std::make_shared<HttpClientJNI>(env))
    , m_javaPlatform(env, platform)
    , m_platformClass(env, env->GetObjectClass(platform))
    , m_capabilitiesClass(env, FindPlayerClass(env, "VideoCapabilities"))
{
    m_capabilities.supportsLowLatencyABR = true;
}

std::unique_ptr<MediaDecoder> PlatformJNI::createDecoder(std::shared_ptr<const MediaFormat> format)
{
    if (!format) {
        return nullptr;
    }

    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (!env) {
        return nullptr;
    }

    jni::LocalRef<jobject> javaFormat(env, MediaDecoderJNI::createMediaFormat(env, *format));
    jni::LocalRef<jobject> decoder(env, env->CallObjectMethod(m_javaPlatform.get(), s_createDecoder, javaFormat.get()));

    checkException(env);

    if (!decoder) {
        return nullptr;
    }

    return std::unique_ptr<MediaDecoder>(new MediaDecoderJNI(env, m_javaPlatform, decoder));
}

std::unique_ptr<MediaRenderer> PlatformJNI::createRenderer(const ReferenceClock& clock, std::shared_ptr<const MediaFormat> format)
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (!env) {
        return nullptr;
    }

    jni::LocalRef<jobject> javaFormat(env, MediaDecoderJNI::createMediaFormat(env, *format));
    jni::LocalRef<jobject> renderer(env, env->CallObjectMethod(m_javaPlatform.get(), s_createRenderer, javaFormat.get()));
    checkException(env);

    if (renderer) {
        if (format->getType().isAudio()) {
            return std::unique_ptr<MediaRendererJNI>(new AudioRendererJNI(env, m_javaPlatform, renderer.get()));
        } else if (format->getType().isVideo()) {
            return std::unique_ptr<MediaRendererJNI>(new VideoRendererJNI(env, m_javaPlatform, renderer.get(), clock));
        }
    }

    return nullptr;
}

std::unique_ptr<DrmSession> PlatformJNI::createDrmSession(const std::vector<uint8_t>& system, DrmSession::Listener& listener)
{
    if (getApiLevel() >= __ANDROID_API_J_MR2__) {
        jni::AttachThread attachThread(jni::getVM());
        JNIEnv* env = attachThread.getEnv();
        return std::unique_ptr<DrmSession>(new DrmSessionJNI(env, system, listener));
    }

    return nullptr;
}

const std::set<MediaType>& PlatformJNI::getSupportedMediaTypes() const
{
    static std::set<MediaType> types = NativePlatform::getSupportedMediaTypes();

    if (!types.count(MediaType::Video_VP9)) {

        jni::AttachThread attachThread(jni::getVM());
        JNIEnv* env = attachThread.getEnv();

        if (env->CallStaticBooleanMethod(m_capabilitiesClass, s_capabilitiesVP9Supported)) {
            types.insert(MediaType::Video_VP9);
        }
    }

    return types;
}

const std::set<std::vector<uint8_t>>& PlatformJNI::getSupportedProtectionSystems() const
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    static std::set<std::vector<uint8_t>> result;

    if (result.empty()) {
        jni::LocalRef<jobjectArray> schemes(env, (jobjectArray)env->CallStaticObjectMethod(m_platformClass.get(), s_getProtectionSystemUUIDs));

        for (int i = 0; i < env->GetArrayLength(schemes.get()); i++) {
            jni::LocalRef<jobject> uuid(env, env->GetObjectArrayElement(schemes.get(), i));
            uint8_t* data = reinterpret_cast<uint8_t*>(env->GetDirectBufferAddress(uuid.get()));

            result.emplace(data, data + env->GetDirectBufferCapacity(uuid.get()));
        }
    }

    return result;
}

VideoDecoderCapabilities PlatformJNI::getVideoDecoderCapabilities(const MediaType& mediaType)
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    jni::LocalRef<jstring> mediaTypeString(env, env->NewStringUTF(mediaType.name.c_str()));
    jni::LocalRef<jobject> capabilities(env, env->CallObjectMethod(m_javaPlatform.get(), s_getVideoDecoderCapabilities, mediaTypeString.get()));
    VideoDecoderCapabilities decoderCapabilities;
    if (capabilities) {
        decoderCapabilities.maxBitrate = env->GetIntField(capabilities, s_capabilitiesMaxBitrate);
        decoderCapabilities.maxFramerate = env->GetIntField(capabilities, s_capabilitiesMaxFramerate);
        decoderCapabilities.maxWidth = env->GetIntField(capabilities, s_capabilitiesMaxWidth);
        decoderCapabilities.maxHeight = env->GetIntField(capabilities, s_capabilitiesMaxHeight);
        decoderCapabilities.maxProfile = env->GetIntField(capabilities, s_capabilitiesMaxProfile);
        decoderCapabilities.maxLevel = env->GetIntField(capabilities, s_capabilitiesMaxLevel);
    }

    return decoderCapabilities;
}

void PlatformJNI::setCurrentThreadName(const std::string& name)
{
    pthread_setname_np(pthread_self(), name.c_str());
}

void PlatformJNI::onThreadCreated(std::thread::id id, const std::string& name)
{
    (void)id;
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (env) {
        jstring threadName = env->NewStringUTF(name.c_str());
        env->CallVoidMethod(m_javaPlatform.get(), s_onThreadCreated, threadName);
        checkException(env);
    }
}

int PlatformJNI::getApiLevel()
{
    static int apiLevel = 0;
    if (apiLevel == 0) {
        // __system_property_get was removed in Android L so have to go through java
        jni::AttachThread attachThread(jni::getVM());
        JNIEnv* env = attachThread.getEnv();
        jni::LocalRef<jclass> buildVersion(env, env->FindClass("android/os/Build$VERSION"));
        jfieldID sdkField = env->GetStaticFieldID(buildVersion.get(), "SDK_INT", "I");
        apiLevel = env->GetStaticIntField(buildVersion.get(), sdkField);
    }
    return apiLevel;
}

void PlatformJNI::checkException(JNIEnv* env)
{
    if (env && env->ExceptionOccurred()) {
        env->ExceptionDescribe();
        env->ExceptionClear();
    }
}
}
