#pragma once

#include "AttachThread.hpp"
#include <jni.h>
#include <string>

// scoped types for use with JNI references
namespace jni {
void setVM(JavaVM* vm);
JavaVM* getVM();

template <typename T, class R>
class ScopedRef {
public:
    ScopedRef()
        : m_ref(nullptr)
        , m_env(nullptr)
    {
    }
    ScopedRef(JNIEnv* env, T object)
        : m_env(env)
    {
        m_ref = R::newRef(m_env, object);
    }
    ScopedRef(JNIEnv* env, std::string className)
        : ScopedRef(env, className.c_str())
    {
    }
    ScopedRef(JNIEnv* env, const char* className)
        : m_env(env)
    {
        m_ref = R::newRef(m_env, m_env->FindClass(className));
    }
    virtual ~ScopedRef() { R::deleteRef(m_env, m_ref); }
    T get() const { return m_ref; }
    operator T() const { return m_ref; }

    ScopedRef(const ScopedRef& other)
    {
        m_env = other.m_env;
        m_ref = R::newRef(m_env, other.m_ref);
    }
    const ScopedRef& operator=(const ScopedRef& other)
    {
        m_env = other.m_env;
        m_ref = R::newRef(m_env, other.m_ref);
        return *this;
    }

private:
    T m_ref;
    JNIEnv* m_env;
};

template <typename T>
class LocalRef : public ScopedRef<T, LocalRef<T>> {
public:
    using ScopedRef<T, LocalRef<T>>::ScopedRef;
    static T newRef(JNIEnv* env, const T ref)
    {
        (void)env;
        return ref;
    }
    static void deleteRef(JNIEnv* env, T ref)
    {
        if (env && ref) {
            env->DeleteLocalRef(ref);
        }
    }
};

template <typename T>
class GlobalRef : public ScopedRef<T, GlobalRef<T>> {
public:
    using ScopedRef<T, GlobalRef<T>>::ScopedRef;
    static T newRef(JNIEnv* env, const T ref)
    {
        if (ref) {
            return static_cast<T>(env->NewGlobalRef(ref));
        }

        return nullptr;
    }
    static void deleteRef(JNIEnv* env, T ref)
    {
        if (ref) {
            jni::AttachThread attachThread(getVM());
            env = attachThread.getEnv();

            if (env) {
                env->DeleteGlobalRef(ref);
            }
        }
    }
};

template <typename T>
class WeakGlobalRef : public ScopedRef<T, WeakGlobalRef<T>> {
public:
    using ScopedRef<T, WeakGlobalRef<T>>::ScopedRef;
    static T newRef(JNIEnv* env, const T ref)
    {
        if (ref) {
            return static_cast<T>(env->NewWeakGlobalRef(ref));
        }

        return nullptr;
    }
    static void deleteRef(JNIEnv* env, T ref)
    {
        if (ref) {
            jni::AttachThread attachThread(getVM());
            env = attachThread.getEnv();

            if (env) {
                env->DeleteWeakGlobalRef(ref);
            }
        }
    }
};

class StringRef {
public:
    StringRef(JNIEnv* env, jstring str)
        : m_env(env)
        , m_jstring(str)
        , m_chars(nullptr)
    {
        m_chars = env->GetStringUTFChars(str, nullptr);

        if (m_chars) {
            m_string = std::string(m_chars);
        }
    }
    virtual ~StringRef()
    {
        if (m_jstring && m_chars) {
            m_env->ReleaseStringUTFChars(m_jstring, m_chars);
        }
    }
    const std::string& get() { return m_string; }
    operator const std::string&() const { return m_string; }
    StringRef(const StringRef& other) = delete;
    const StringRef& operator=(const StringRef& other) = delete;

private:
    JNIEnv* m_env;
    jstring m_jstring;
    const char* m_chars;
    std::string m_string;
};
}
