#include "player_jni.hpp"
#include "LogCat.hpp"
#include "MediaDecoderJNI.hpp"
#include "MediaRendererJNI.hpp"
#include "debug/trace.hpp"
#include "http_jni.hpp"
#include "package.hpp"

namespace twitch {

#define JNI_METHOD(METHOD) JNICALL Java_tv_twitch_android_player_MediaPlayer_##METHOD

void JNIWrapper::initialize(JNIEnv* env)
{
    jclass playerClass = FindPlayerClass(env, "MediaPlayer");
    s_playerHandleError = env->GetMethodID(playerClass, "handleError", "(IIILjava/lang/String;)V");
    s_playerHandleQualityChange = env->GetMethodID(playerClass, "handleQualityChange", ("(L" + PlayerPackage + "Quality;)V").c_str());
    s_playerHandleRebuffering = env->GetMethodID(playerClass, "handleRebuffering", "()V");
    s_playerHandleStateChange = env->GetMethodID(playerClass, "handleStateChange", "(I)V");
    s_playerHandleMetadata = env->GetMethodID(playerClass, "handleMetadata", "(Ljava/lang/String;Ljava/nio/ByteBuffer;)V");
    s_playerHandleAnalytics = env->GetMethodID(playerClass, "handleAnalyticsEvent", "(Ljava/lang/String;Ljava/lang/String;)V");
    jclass qualityClass = FindPlayerClass(env, "Quality");
    s_playerInitQuality = env->GetMethodID(qualityClass, "<init>", "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;III)V");

    jclass statisticsClass = FindPlayerClass(env, "Statistics");
    s_statisticsBitRate = env->GetFieldID(statisticsClass, "bitRate", "I");
    s_statisticsFrameRate = env->GetFieldID(statisticsClass, "frameRate", "I");
    s_statisticsDecodedFrames = env->GetFieldID(statisticsClass, "decodedFrames", "I");
    s_statisticsDroppedFrames = env->GetFieldID(statisticsClass, "droppedFrames", "I");
    s_statisticsRenderedFrames = env->GetFieldID(statisticsClass, "renderedFrames", "I");
}

JNIWrapper::JNIWrapper(JNIEnv* env, jobject player, jobject platform, jstring tempPath)
    : m_vmThread(jni::getVM())
    , m_object(env, player)
    , m_qualityClass(env, FindPlayerClass(env, "Quality"))
{
    auto androidPlatform = std::make_shared<PlatformJNI>(env, platform);
    m_player = std::make_shared<MediaPlayer>(*this, androidPlatform);
    jni::StringRef tempPathString(env, tempPath);
    androidPlatform->setTempPath(tempPathString.get());
}

JNIWrapper::~JNIWrapper()
{
    m_player.reset(); // ensures no callbacks are run
}

void JNIWrapper::onError(const Error& error)
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (env) {
        jni::LocalRef<jstring> javaMessage(env, env->NewStringUTF(error.message.c_str()));
        env->CallVoidMethod(m_object, s_playerHandleError,
            static_cast<jint>(error.source),
            static_cast<jint>(error.result.value), error.result.code, javaMessage.get());
    }
}

void JNIWrapper::onDurationChanged(MediaTime duration)
{
    (void)duration;
}

void JNIWrapper::onRebuffering()
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (env) {
        env->CallVoidMethod(m_object, s_playerHandleRebuffering);
    }
}

void JNIWrapper::onRecoverableError(const Error& error)
{
    (void)error;
}

void JNIWrapper::onSessionData(const std::map<std::string, std::string>& properties)
{
    (void)properties; // TODO
}

void JNIWrapper::onStateChanged(MediaPlayer::State state)
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (env) {
        env->CallVoidMethod(m_object, s_playerHandleStateChange, static_cast<jint>(state));
    }
}

void JNIWrapper::onMetadata(const std::string& type, const std::vector<uint8_t>& data)
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (env) {
        jni::LocalRef<jstring> typeRef(env, env->NewStringUTF(type.c_str()));
        uint8_t* bufferPtr = const_cast<uint8_t*>(data.data());
        jni::LocalRef<jobject> buffer(env, env->NewDirectByteBuffer(bufferPtr, data.size()));
        env->CallVoidMethod(m_object, s_playerHandleMetadata, typeRef.get(), buffer.get());
    }
}

void JNIWrapper::onPositionChanged(MediaTime position)
{
    (void)position;
}

void JNIWrapper::onQualityChanged(const Quality& quality)
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (env) {
        jni::LocalRef<jobject> qualityObject(env, createQuality(env, quality));
        env->CallVoidMethod(m_object, s_playerHandleQualityChange, qualityObject.get());
    }
}

void JNIWrapper::onSeekCompleted(MediaTime time)
{
    (void)time;
}

void JNIWrapper::onAnalyticsEvent(const std::string& name, const std::string& properties)
{
    jni::AttachThread attachThread(jni::getVM());
    JNIEnv* env = attachThread.getEnv();

    if (env) {
        jni::LocalRef<jstring> nameRef(env, env->NewStringUTF(name.c_str()));
        jni::LocalRef<jstring> propertiesRef(env, env->NewStringUTF(properties.c_str()));
        env->CallVoidMethod(m_object, s_playerHandleAnalytics, nameRef.get(), propertiesRef.get());
    }
}

jobject JNIWrapper::createQuality(JNIEnv* env, const Quality& quality)
{
    jni::LocalRef<jstring> nameRef(env, env->NewStringUTF(quality.name.c_str()));
    jni::LocalRef<jstring> typeRef(env, env->NewStringUTF(quality.type.c_str()));
    jni::LocalRef<jstring> codecsRef(env, env->NewStringUTF(quality.codecs.c_str()));
    return env->NewObject(m_qualityClass.get(), s_playerInitQuality,
        nameRef.get(),
        typeRef.get(),
        codecsRef.get(),
        static_cast<jint>(quality.bitrate),
        static_cast<jint>(quality.width),
        static_cast<jint>(quality.height));
}

jmethodID JNIWrapper::s_playerHandleError;
jmethodID JNIWrapper::s_playerHandleQualityChange;
jmethodID JNIWrapper::s_playerHandleRebuffering;
jmethodID JNIWrapper::s_playerHandleStateChange;
jmethodID JNIWrapper::s_playerHandleMetadata;
jmethodID JNIWrapper::s_playerHandleAnalytics;
jmethodID JNIWrapper::s_playerInitQuality;
jfieldID JNIWrapper::s_statisticsBitRate;
jfieldID JNIWrapper::s_statisticsFrameRate;
jfieldID JNIWrapper::s_statisticsDecodedFrames;
jfieldID JNIWrapper::s_statisticsDroppedFrames;
jfieldID JNIWrapper::s_statisticsRenderedFrames;

template <typename Method, typename... Args>
void invoke(jlong ptr, Method m, Args&&... args)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);
    if (wrapper) {
        MediaPlayer* player = wrapper->m_player.get();
        (player->*m)(std::forward<Args>(args)...);
    }
}

extern "C" {
JNIEXPORT void JNI_METHOD(initialize)(JNIEnv* env)
{
    JNIWrapper::initialize(env);
}

JNIEXPORT jlong JNI_METHOD(init)(JNIEnv* env, jobject player, jobject platform, jstring tempPath)
{
    JNIWrapper* wrapper = new JNIWrapper(env, player, platform, tempPath);
    return reinterpret_cast<jlong>(wrapper);
}

JNIEXPORT void JNI_METHOD(load)(JNIEnv* env, jobject object, jlong ptr, jstring path, jstring mediaType)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper && path) {
        wrapper->m_player->load(jni::StringRef(env, path), jni::StringRef(env, mediaType));
    }
}

JNIEXPORT void JNI_METHOD(play)(JNIEnv* env, jobject object, jlong ptr)
{
    invoke(ptr, &MediaPlayer::play);
}

JNIEXPORT void JNI_METHOD(pause)(JNIEnv* env, jobject object, jlong ptr)
{
    invoke(ptr, &MediaPlayer::pause);
}

JNIEXPORT jboolean JNI_METHOD(isSeekable)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);
    return wrapper && wrapper->m_player->isSeekable() ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT void JNI_METHOD(seekTo)(JNIEnv* env, jobject object, jlong ptr, jlong position)
{
    invoke(ptr, &MediaPlayer::seekTo, MediaTime(position, std::milli::den));
}

JNIEXPORT void JNI_METHOD(release)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper) {
        delete wrapper;
    }
}

JNIEXPORT jboolean JNI_METHOD(isMuted)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);
    return wrapper && wrapper->m_player->isMuted();
}

JNIEXPORT jint JNI_METHOD(getState)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper) {
        Player::State state = wrapper->m_player->getState();
        return static_cast<int>(state);
    }

    return -1;
}

JNIEXPORT jlong JNI_METHOD(getPosition)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper) {
        MediaTime time = wrapper->m_player->getPosition();
        return time.milliseconds().count();
    }

    return -1;
}

JNIEXPORT jlong JNI_METHOD(getBufferedPosition)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper) {
        MediaTime time = wrapper->m_player->getBufferedPosition();
        return time.milliseconds().count();
    }

    return -1;
}

JNIEXPORT jlong JNI_METHOD(getDuration)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper) {
        MediaTime time = wrapper->m_player->getDuration();
        return time == MediaTime::max() ? -1 : time.milliseconds().count();
    }

    return -1;
}

JNIEXPORT jobject JNI_METHOD(getQualities)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (!wrapper) {
        return nullptr;
    }

    jclass hashSetClass = env->FindClass("java/util/HashSet");
    jmethodID hashSetInit = env->GetMethodID(hashSetClass, "<init>", "()V");
    jmethodID addMethod = env->GetMethodID(hashSetClass, "add", "(Ljava/lang/Object;)Z");
    jobject hashSet = env->NewObject(hashSetClass, hashSetInit);

    for (const auto& quality : wrapper->m_player->getQualities()) {
        jobject item = wrapper->createQuality(env, quality);
        env->CallBooleanMethod(hashSet, addMethod, item);
    }

    return hashSet;
}

JNIEXPORT jobject JNI_METHOD(getQuality)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (!wrapper) {
        return nullptr;
    }

    return wrapper->createQuality(env, wrapper->m_player->getQuality());
}

JNIEXPORT void JNI_METHOD(setQuality)(JNIEnv* env, jobject object, jlong ptr, jstring value, jboolean adaptive)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper) {
        jni::StringRef name(env, value);

        for (const auto& quality : wrapper->m_player->getQualities()) {
            if (quality.name == name.get()) {
                wrapper->m_player->setQuality(quality, adaptive);
                break;
            }
        }
    }
}

JNIEXPORT jboolean JNI_METHOD(getAutoSwitchQuality)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);
    return wrapper && wrapper->m_player->getAutoSwitchQuality();
}

JNIEXPORT void JNI_METHOD(setAutoSwitchQuality)(JNIEnv* env, jobject object, jlong ptr, jboolean value)
{
    invoke(ptr, &MediaPlayer::setAutoSwitchQuality, value == JNI_TRUE);
}

JNIEXPORT void JNI_METHOD(setAutoInitialBitrate)(JNIEnv* env, jobject object, jlong ptr, jint bitrate)
{
    invoke(ptr, &MediaPlayer::setAutoInitialBitrate, bitrate);
}

JNIEXPORT void JNI_METHOD(setAutoMaxBitrate)(JNIEnv* env, jobject object, jlong ptr, jint bitrate)
{
    invoke(ptr, &MediaPlayer::setAutoMaxBitrate, bitrate);
}

JNIEXPORT void JNI_METHOD(setAutoMaxVideoSize)(JNIEnv* env, jobject object, jlong ptr, jint width, jint height)
{
    invoke(ptr, &MediaPlayer::setAutoMaxVideoSize, width, height);
}

JNIEXPORT void JNI_METHOD(setLiveLowLatencyEnabled)(JNIEnv* env, jobject object, jlong ptr, jboolean enabled)
{
    invoke(ptr, &MediaPlayer::setLiveLowLatencyEnabled, enabled == JNI_TRUE);
}

JNIEXPORT void JNI_METHOD(setLooping)(JNIEnv* env, jobject object, jlong ptr, jboolean loop)
{
    invoke(ptr, &MediaPlayer::setLooping, loop);
}

JNIEXPORT void JNI_METHOD(setMinBuffer)(JNIEnv* env, jobject object, jlong ptr, jlong duration)
{
    invoke(ptr, &MediaPlayer::setMinBuffer, MediaTime(duration, std::milli::den));
}

JNIEXPORT void JNI_METHOD(setMaxBuffer)(JNIEnv* env, jobject object, jlong ptr, jlong duration)
{
    invoke(ptr, &MediaPlayer::setMaxBuffer, MediaTime(duration, std::milli::den));
}

JNIEXPORT void JNI_METHOD(setMuted)(JNIEnv* env, jobject object, jlong ptr, jboolean muted)
{
    invoke(ptr, &MediaPlayer::setMuted, muted);
}

JNIEXPORT void JNI_METHOD(setSurface)(JNIEnv* env, jobject object, jlong ptr, jobject surface)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);
    wrapper->m_surface = jni::GlobalRef<jobject>(env, surface);
    invoke(ptr, &MediaPlayer::setSurface, wrapper->m_surface);
}

JNIEXPORT void JNI_METHOD(setPlaybackRate)(JNIEnv* env, jobject object, jlong ptr, jfloat rate)
{
    invoke(ptr, &MediaPlayer::setPlaybackRate, rate);
}

JNIEXPORT void JNI_METHOD(setVolume)(JNIEnv* env, jobject object, jlong ptr, jfloat volume)
{
    invoke(ptr, &MediaPlayer::setVolume, volume);
}

JNIEXPORT void JNI_METHOD(setClientId)(JNIEnv* env, jobject object, jlong ptr, jstring id)
{
    jni::StringRef idRef(env, id);
    invoke(ptr, &MediaPlayer::setClientId, idRef.get());
}

JNIEXPORT void JNI_METHOD(setDeviceId)(JNIEnv* env, jobject object, jlong ptr, jstring id)
{
    jni::StringRef idRef(env, id);
    invoke(ptr, &MediaPlayer::setDeviceId, idRef.get());
}

JNIEXPORT jlong JNI_METHOD(getAverageBitrate)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);
    if (wrapper) {
        return static_cast<jlong>(wrapper->m_player->getAverageBitrate());
    }
    return 0;
}

JNIEXPORT jlong JNI_METHOD(getBandwidthEstimate)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);
    if (wrapper) {
        return static_cast<jlong>(wrapper->m_player->getBandwidthEstimate());
    }
    return 0;
}

JNIEXPORT jlong JNI_METHOD(getLiveLatency)(JNIEnv* env, jobject object, jlong ptr)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper) {
        MediaTime time = wrapper->m_player->getLiveLatency();
        return time.milliseconds().count();
    }

    return -1;
}

JNIEXPORT void JNI_METHOD(getStatistics)(JNIEnv* env, jobject object, jlong ptr, jobject statisticsObject)
{
    JNIWrapper* wrapper = reinterpret_cast<JNIWrapper*>(ptr);

    if (wrapper) {
        const auto& statistics = wrapper->m_player->getStatistics();
        env->SetIntField(statisticsObject, wrapper->s_statisticsBitRate, statistics.getBitRate());
        env->SetIntField(statisticsObject, wrapper->s_statisticsFrameRate, statistics.getFrameRate());
        env->SetIntField(statisticsObject, wrapper->s_statisticsDecodedFrames, statistics.getDecodedFrames());
        env->SetIntField(statisticsObject, wrapper->s_statisticsDroppedFrames, statistics.getDroppedFrames());
        env->SetIntField(statisticsObject, wrapper->s_statisticsRenderedFrames, statistics.getRenderedFrames());
    }
}
}
}
