package tv.twitch.android.player;

import android.annotation.SuppressLint;
import android.media.AudioAttributes;
import android.media.AudioFormat;
import android.media.AudioManager;
import android.media.AudioTrack;
import android.media.MediaFormat;
import android.media.PlaybackParams;
import android.os.Build;
import android.os.Process;
import android.util.Log;

import java.nio.ByteBuffer;

import static android.media.AudioFormat.CHANNEL_OUT_5POINT1;
import static android.media.AudioFormat.CHANNEL_OUT_7POINT1_SURROUND;
import static android.media.AudioFormat.CHANNEL_OUT_BACK_CENTER;
import static android.media.AudioFormat.CHANNEL_OUT_DEFAULT;
import static android.media.AudioFormat.CHANNEL_OUT_FRONT_CENTER;
import static android.media.AudioFormat.CHANNEL_OUT_MONO;
import static android.media.AudioFormat.CHANNEL_OUT_QUAD;
import static android.media.AudioFormat.CHANNEL_OUT_STEREO;

/**
 * Plays decoded audio samples onto a {@link AudioTrack}.
 *
 * @author Nikhil Purushe
 */
class AudioTrackRenderer implements MediaRenderer, AudioRenderer {

    private static final int BUFFERS = 2;

    private final int streamType;
    private final int mode;
    private AudioTrack track;
    private byte[] output;
    private int audioSessionId;
    private int frameSizeBytes;
    private long framesWritten;
    private long playheadBasePosition;
    private long playheadPrevPosition;
    private int bytesWritten;
    private int bufferSize;
    private long sampleTimeUs;
    private float volume;
    private float playbackRate;
    private boolean started;

    AudioTrackRenderer() {
        this(AudioTrack.MODE_STREAM, AudioManager.STREAM_MUSIC);
    }

    AudioTrackRenderer(int mode, int streamType) {
        this.mode = mode;
        this.streamType = streamType;
        this.volume = 1.0f;
        this.playbackRate = 1.0f;
        Process.setThreadPriority(Process.THREAD_PRIORITY_URGENT_AUDIO);
    }

    private void updateTrackVolume() {
        if (track != null) {
            Log.i(Logging.TAG, "Set audio track volume " + volume);
            int result;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                result = track.setVolume(volume);
            } else {
                result = track.setStereoVolume(volume, volume);
            }
            checkResult(result);
        }
    }

    private void updatePlaybackRate() {
        if (track != null && android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M
                && track.getState() == AudioTrack.STATE_INITIALIZED) {
            Log.i(Logging.TAG, "Set playback rate " + playbackRate);
            PlaybackParams params = new PlaybackParams();
            track.setPlaybackParams(params.setSpeed(playbackRate));
        }
    }

    @Override
    public int getAudioSessionId() {
        return audioSessionId;
    }

    @Override
    public void setVolume(float volume) {
        this.volume = volume;
        updateTrackVolume();
    }

    @Override
    public void setPlaybackRate(float rate) {
        // playback rate cannot exceed pcm buffer multiplier
        if (rate > BUFFERS) {
            rate = BUFFERS;
        }
        playbackRate = rate;
        updatePlaybackRate();
    }

    @SuppressLint("InlinedApi")
    private static int createChannelConfig(int channelCount) {
        // see https://developer.android.com/reference/android/media/AudioFormat.html
        switch (channelCount) {
            case 1:
                return CHANNEL_OUT_MONO;
            case 2:
                return CHANNEL_OUT_STEREO;
            case 3:
                return CHANNEL_OUT_STEREO | CHANNEL_OUT_FRONT_CENTER;
            case 4:
                return CHANNEL_OUT_QUAD;
            case 5:
                return CHANNEL_OUT_QUAD | CHANNEL_OUT_FRONT_CENTER;
            case 6:
                return CHANNEL_OUT_5POINT1;
            case 7:
                return CHANNEL_OUT_5POINT1 | CHANNEL_OUT_BACK_CENTER;
            case 8:
                return CHANNEL_OUT_7POINT1_SURROUND;
            default:
                return CHANNEL_OUT_DEFAULT;
        }
    }

    @Override
    public void configure(MediaFormat format) {
        int sampleRate = format.getInteger(MediaFormat.KEY_SAMPLE_RATE);
        int channelCount = format.getInteger(MediaFormat.KEY_CHANNEL_COUNT);

        if (track != null) {
            // check if new track not needed
            if (track.getSampleRate() == sampleRate && track.getChannelCount() == channelCount) {
                return;
            }
            release();
        }

        int encoding = AudioFormat.ENCODING_PCM_16BIT;
        int channelConfig = createChannelConfig(channelCount);
        bufferSize = AudioTrack.getMinBufferSize(sampleRate, channelConfig, encoding);
        if (bufferSize == AudioTrack.ERROR_BAD_VALUE) {
            bufferSize = sampleRate;
            Log.w(Logging.TAG, "Error getting min buffer size using sample rate " + sampleRate);
        }
        bufferSize *= BUFFERS;
        frameSizeBytes = channelCount * 2; // 16 bit
        framesWritten = 0;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            AudioAttributes audioAttributes = new AudioAttributes.Builder()
                    .setContentType(AudioAttributes.CONTENT_TYPE_MOVIE)
                    .setUsage(AudioAttributes.USAGE_MEDIA)
                    .build();

            AudioFormat audioFormat = new AudioFormat.Builder()
                    .setEncoding(encoding)
                    .setSampleRate(sampleRate)
                    .setChannelMask(channelConfig)
                    .build();

            Log.w(Logging.TAG, "Creating audio track format: " + audioFormat);
            track = new AudioTrack(audioAttributes, audioFormat, bufferSize, mode,
                    audioSessionId > 0 ? audioSessionId : AudioManager.AUDIO_SESSION_ID_GENERATE);
        } else {
            track = new AudioTrack(streamType, sampleRate, channelConfig, encoding, bufferSize, mode,
                    audioSessionId > 0 ? audioSessionId : AudioManager.AUDIO_SESSION_ID_GENERATE);
        }

        audioSessionId = track.getAudioSessionId();
        updateTrackVolume();
        updatePlaybackRate();
    }

    @Override
    public void start() {
        started = true;
    }

    @Override
    public void stop() {
        started = false;
        if (track != null && track.getPlayState() == AudioTrack.PLAYSTATE_PLAYING) {
            track.pause();
        }
    }

    @Override
    public void flush() {
        sampleTimeUs = 0;
        framesWritten = 0;
        bytesWritten = 0;
        playheadPrevPosition = 0;

        if (track != null) {
            track.stop();
            track.flush();
            long position = track.getPlaybackHeadPosition();
            // release track if playhead not at zero or < 4.4 since flush may not work
            if (position > 0 || Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) {
                Log.w(Logging.TAG, "audio position at flush " + position);
                release();
            }
        }
    }

    @Override
    public void render(ByteBuffer buffer, int size, long presentationTimeUs) {
        if (track == null || size == 0) {
            return;
        }

        int result;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            result = track.write(buffer, size, AudioTrack.WRITE_BLOCKING);
        } else {
            if (buffer.isDirect()) {
                if (output == null || output.length < size) {
                    output = new byte[size];
                }
                buffer.get(output, 0, size);
                result = track.write(output, 0, size);
            } else {
                result = track.write(buffer.array(), 0, size);
            }
        }

        framesWritten += size / frameSizeBytes;
        bytesWritten += size;
        sampleTimeUs = presentationTimeUs;

        if (started && bytesWritten >= bufferSize && track.getPlayState() != AudioTrack.PLAYSTATE_PLAYING) {
            track.play();
        }

        checkResult(result);
    }

    @Override
    public void release() {
        flush();
        if (track != null) {
            try {
                track.release();
            } finally {
                track = null;
            }
        }
    }

    @Override
    public long getRenderedPresentationTime() {
        if (sampleTimeUs == 0 || track == null || track.getPlayState() != AudioTrack.PLAYSTATE_PLAYING) {
            return -1;
        }
        long position = 0xFFFFFFFFL & track.getPlaybackHeadPosition();

        // detect rollover in getPlaybackHeadPosition
        if (playheadPrevPosition > 0 && position < playheadPrevPosition) {
            Log.w(Logging.TAG, "AudioTrack.getPlaybackHeadPosition "
                    + position + " < previous " + playheadPrevPosition);
            playheadBasePosition = playheadPrevPosition;
        }
        playheadPrevPosition = position;
        // calculate total pts
        position += playheadBasePosition;
        long frames = framesWritten - position;

        long audioLatencyUs = (long) ((frames / (double)track.getSampleRate()) * 1000000);
        return sampleTimeUs - audioLatencyUs;
    }

    private void checkResult(int result) {
        switch (result) {
            case AudioTrack.ERROR_INVALID_OPERATION:
                Log.e(Logging.TAG, "AudioTrack.ERROR_INVALID_OPERATION");
                break;
            case AudioTrack.ERROR_BAD_VALUE:
                Log.e(Logging.TAG, "AudioTrack.ERROR_BAD_VALUE");
                break;
            case AudioTrack.ERROR_DEAD_OBJECT:
                Log.e(Logging.TAG, "AudioTrack.ERROR_DEAD_OBJECT");
                flush();
                break;
            default:
            case AudioTrack.SUCCESS:
                break;
        }
    }
}
