package tv.twitch.android.player;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

/**
 * Reads Protection System Specific Header (PSSH) info
 */
class DrmPssh {
    private static final int PSSH = Charset.forName("UTF-8").encode("pssh").getInt();

    private UUID systemId;
    private List<byte[]> keyIds = new ArrayList<>();
    private byte[] data = new byte[0];

    DrmPssh(ByteBuffer buffer) {
        buffer = buffer.duplicate().order(ByteOrder.BIG_ENDIAN);
        byte[] uuidBytes = new byte[16];

        if (buffer.remaining() >= 12 + uuidBytes.length) {
            int size = buffer.getInt();
            int type = buffer.getInt();
            if (type == PSSH && buffer.remaining() >= size - 8) {
                int flags = buffer.getInt();
                int version = flags >> 24;
                buffer.get(uuidBytes);
                systemId = getUUID(ByteBuffer.wrap(uuidBytes));

                if (version > 0) {
                    int keyCount = buffer.getInt();
                    for (int i = 0; i < keyCount; i++) {
                        byte[] keyId = new byte[16];
                        buffer.get(keyId);
                        keyIds.add(keyId);
                    }
                }

                int dataSize = buffer.getInt();
                if (dataSize > 0) {
                    data = new byte[dataSize];
                    buffer.get(data, 0, data.length);
                }
            }
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof DrmPssh) {
            DrmPssh other = (DrmPssh) obj;
            return systemId != null && systemId.equals(other.systemId)
                    && keyIds.equals(other.keyIds)
                    && Arrays.equals(data, other.data);
        }
        return false;
    }

    @Override
    public int hashCode() {
        if (data.length > 0) {
            return Arrays.hashCode(data);
        } else {
            return keyIds.hashCode();
        }
    }

    private static UUID getUUID(ByteBuffer buffer) {
        long mostSigBits = buffer.getLong();
        long leastSigBits = buffer.getLong();
        return new UUID(mostSigBits, leastSigBits);
    }

    public UUID getSystemId() {
        return systemId;
    }

    public List<byte[]> getKeyIds() {
        return keyIds;
    }

    public byte[] getData() {
        return data;
    }
}
