package tv.twitch.android.player;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.media.DeniedByServerException;
import android.media.MediaCrypto;
import android.media.MediaCryptoException;
import android.media.MediaDrm;
import android.media.MediaFormat;
import android.media.NotProvisionedException;
import android.media.UnsupportedSchemeException;
import android.os.Build;
import android.util.Log;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Android implementation of a DRM session using {@link MediaDrm}. Wraps all calls to the
 * {@link MediaDrm} provides a listener callback interface mapping {@link MediaDrm.OnEventListener}
 * events to an implementation.
 *
 * @author Nikhil Purushe
 */
@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
@SuppressWarnings("unused") // called from native
public class DrmSession implements Releasable, MediaDrm.OnEventListener {

    public interface Listener {
        void onKeyExpired();
        void onKeyRequested();
        void onProvisionRequested();
        void onError(String message);
    }

    private static final Map<UUID, DrmSession> sessions = new ConcurrentHashMap<>();

    private final UUID systemUUID;
    private final Listener listener;
    private final MediaDrm drm;
    private final Map<DrmPssh, MediaCrypto> mediaCrypto;
    private byte[] session;
    private boolean sessionSharing;

    public static class OpaqueRequest {
        String url;
        byte[] data;
    }

    public static DrmSession create(UUID systemUUID, Listener listener) {
        if (MediaDrm.isCryptoSchemeSupported(systemUUID)) {
            return new DrmSession(systemUUID, listener);
        }
        return null;
    }

    // static creation method specifically for native code only
    public static DrmSession create(ByteBuffer systemId, long ptr) {
        return create(getUUID(systemId), new DrmListener(ptr));
    }

    public static DrmSession get(MediaFormat format) {
        for (int i = 0;; i++) {
            ByteBuffer protectionData = format.getByteBuffer("drm-" + i);
            if (protectionData == null) {
                break;
            }
            DrmPssh pssh = new DrmPssh(protectionData);
            UUID systemId = pssh.getSystemId();
            if (systemId != null && sessions.containsKey(systemId)) {
                return sessions.get(systemId);
            }
        }

        return null;
    }

    @SuppressLint("UseSparseArrays")
    private DrmSession(UUID systemUUID, Listener listener) {
        this.systemUUID = systemUUID;
        this.listener = listener;
        this.mediaCrypto = new HashMap<>();
        try {
            drm = new MediaDrm(systemUUID);
        } catch (UnsupportedSchemeException e) {
            // shouldn't happen since the scheme is checked in advance with isCryptoSchemeSupported
            e.printStackTrace();
            throw new RuntimeException(e);
        }
        drm.setOnEventListener(this);
        // enable session sharing for key rotation
        // https://storage.googleapis.com/wvdocs/Widevine_DRM_Android_Using_Key_Rotation.pdf
        if (systemUUID.equals(ProtectionSystem.WIDEVINE.getUUID())) {
            if (sessionSharing) {
                try {
                    drm.setPropertyString("sessionSharing", "enable");
                } catch (IllegalStateException se) {
                    se.printStackTrace();
                }
            }
        }
        sessions.put(systemUUID, this);
    }

    private void createSession(boolean fromProvisioning) {
        // close previous session if any
        if (session != null) {
            drm.closeSession(session);
            session = null;
        }
        // create the session
        try {
            session = drm.openSession();
        } catch (NotProvisionedException e) {
            e.printStackTrace();
            if (!fromProvisioning) {
                listener.onProvisionRequested();
            } else {
                listener.onError(e.getMessage());
            }
        } catch (Exception e) {
            e.printStackTrace();
            listener.onError(e.getMessage());
        }
    }

    public void initialize() {
        createSession(false);
    }

    @Override
    public void release() {
        if (drm != null) {
            try {
                if (session != null) {
                    drm.closeSession(session);
                }
                synchronized (mediaCrypto) {
                    mediaCrypto.clear();
                }
            } finally {
                drm.release();
                sessions.remove(systemUUID);
            }
        }
    }

    MediaCrypto getMediaCrypto(MediaFormat format) {
        // find matching protection data
        DrmPssh pssh = null;
        for (int i = 0;; i++) {
            ByteBuffer protectionData = format.getByteBuffer("drm-" + i);
            if (protectionData == null) {
                break;
            }
            pssh = new DrmPssh(protectionData);
            if (systemUUID.equals(pssh.getSystemId())) {
                break;
            }
        }
        if (pssh != null) {
            synchronized (mediaCrypto) {
                if (mediaCrypto.containsKey(pssh)) {
                    return mediaCrypto.get(pssh);
                } else {
                    try {
                        MediaCrypto crypto = new MediaCrypto(systemUUID, session);
                        mediaCrypto.put(pssh, crypto);
                        return crypto;
                    } catch (MediaCryptoException e) {
                        e.printStackTrace();
                        listener.onError(e.getMessage());
                    }
                }
            }
        }
        return null;
    }

    public byte[] getSessionId() {
        return session;
    }

    @Override
    public void onEvent(MediaDrm md, byte[] sessionId, int event, int extra, byte[] data) {
        Log.w(Logging.TAG, "MediaDrm event " + event + " extra " + extra);
        switch (event) {
            case MediaDrm.EVENT_KEY_EXPIRED:
                listener.onKeyExpired();
                break;
            case MediaDrm.EVENT_KEY_REQUIRED:
                listener.onKeyRequested();
                break;
            case MediaDrm.EVENT_PROVISION_REQUIRED:
                // deprecated in API 23
                //listener.onProvisionRequested();
                break;
            case MediaDrm.EVENT_SESSION_RECLAIMED:
                break;
            case MediaDrm.EVENT_VENDOR_DEFINED:
                break;
        }
    }

    public OpaqueRequest generateKeyRequest(byte[] init) {
        try {
            // on versions before L init data is the pssh scheme specific data not the pssh box
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
                DrmPssh pssh = new DrmPssh(ByteBuffer.wrap(init));
                init = pssh.getData();
            }

            MediaDrm.KeyRequest keyRequest =
                    drm.getKeyRequest(session, init, "cenc", MediaDrm.KEY_TYPE_STREAMING, null);
            OpaqueRequest request = new OpaqueRequest();
            request.url = keyRequest.getDefaultUrl();
            request.data = keyRequest.getData();
            return request;
        } catch (NotProvisionedException e) {
            listener.onProvisionRequested();
            e.printStackTrace();
        }
        return null;
    }

    public OpaqueRequest generateProvisionRequest() {
        MediaDrm.ProvisionRequest provisionRequest = drm.getProvisionRequest();
        OpaqueRequest request = new OpaqueRequest();
        request.url = provisionRequest.getDefaultUrl();
        request.data = provisionRequest.getData();
        return request;
    }

    public void updateKeyResponse(byte[] data) {
        try {
            drm.provideKeyResponse(session, data);
        } catch (NotProvisionedException e) {
            e.printStackTrace();
            listener.onProvisionRequested();
        } catch (DeniedByServerException e) {
            e.printStackTrace();
            listener.onError(e.getMessage());
        }
    }

    public void updateProvisionResponse(byte[] data) {
        try {
            drm.provideProvisionResponse(data);
        } catch (DeniedByServerException e) {
            e.printStackTrace();
            listener.onError(e.getMessage());
            return;
        }
        // create the session if it failed to be created originally because of provisioning
        if (session == null) {
            createSession(true);
        }
    }

    private static UUID getUUID(ByteBuffer buffer) {
        long mostSigBits = buffer.getLong();
        long leastSigBits = buffer.getLong();
        return new UUID(mostSigBits, leastSigBits);
    }

    private void unprovision() {
        // see https://storage.googleapis.com/wvdocs/Widevine_DRM_Android_Vendor_Extensions.pdf
        try {
            drm.provideProvisionResponse("unprovision".getBytes());
        } catch (Exception e) {
            Log.w(Logging.TAG, "Un-provision failed", e);
        }
    }

    private void setWidevineSecurityLevel(String level) {
        if (systemUUID.equals(ProtectionSystem.WIDEVINE.getUUID())) {
            final String SECURITY_LEVEL = "securityLevel";
            @SuppressLint("WrongConstant")
            String currentLevel = drm.getPropertyString(SECURITY_LEVEL);
            if (!currentLevel.equals(level)) {
                drm.setPropertyString(SECURITY_LEVEL, level);
            }
        }
    }
}
