package tv.twitch.android.player;

import android.media.MediaFormat;
import android.os.Build;

import java.nio.ByteBuffer;
import java.util.Arrays;

/**
 * Support for class for creating MediaFormat instances.
 *
 * @author Nikhil Purushe
 */
@SuppressWarnings("unused") // called from native
class Formats {

    private Formats() {
    }

    private static final byte[] NAL_START_CODE = {0x00, 0x00, 0x00, 0x01};

    private static ByteBuffer addNALStartCodes(ByteBuffer csd) {
        if (csd == null) {
            csd = ByteBuffer.allocateDirect(4);
        }
        if (csd.remaining() > 4) {
            byte[] start = new byte[4];
            csd.get(start);
            csd.rewind();
            // if already has the start code skip it
            if (Arrays.equals(start, NAL_START_CODE)) {
                return csd;
            }
        }
        ByteBuffer coded = ByteBuffer.allocateDirect(csd.remaining() + 4);
        coded.put(NAL_START_CODE);
        coded.put(csd);
        coded.rewind();
        return coded;
    }

    public static MediaFormat createVideoFormat(String mediaType, ByteBuffer[] csd,
                                                int width, int height,
                                                ByteBuffer[] protectionData) {
        mediaType = MediaType.removeParameters(mediaType);
        MediaFormat format = MediaFormat.createVideoFormat(mediaType, width, height);
        // work around configure() bug on Samsung
        format.setInteger(MediaFormat.KEY_MAX_INPUT_SIZE, 0);
        if (csd != null) {
            for (int i = 0; i < csd.length; i++) {
                csd[i] = addNALStartCodes(csd[i]);
                if (csd[i].remaining() > 4) {
                    format.setByteBuffer("csd-" + i, csd[i]);
                }
            }
        }

        setProtectionData(format, protectionData);
        return format;
    }

    public static MediaFormat createAudioFormat(String mediaType, ByteBuffer[] csd,
                                                int sampleRate, int channels,
                                                ByteBuffer[] protectionData) {
        mediaType = MediaType.removeParameters(mediaType);
        if (MediaType.AUDIO_AAC.equals(mediaType)) {
            mediaType = "audio/mp4a-latm";// MediaFormat.MIMETYPE_AUDIO_AAC;
        }
        MediaFormat format = MediaFormat.createAudioFormat(mediaType, sampleRate, channels);
        if (csd != null) {
            for (int i = 0; i < csd.length; i++) {
                format.setByteBuffer("csd-" + i, csd[i]);
            }
        }

        setProtectionData(format, protectionData);
        return format;
    }

    private static void setProtectionData(MediaFormat format, ByteBuffer[] protectionData) {
        if (protectionData != null) {
            for (int i = 0; i < protectionData.length; i++) {
                format.setByteBuffer("drm-" + i, protectionData[i]);
            }
        }
    }

    public static MediaFormat createTextFormat(String mediaType) {
        mediaType = MediaType.removeParameters(mediaType);
        MediaFormat format;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            format = MediaFormat.createSubtitleFormat(mediaType, null);
        } else {
            format = new MediaFormat();
            format.setString(MediaFormat.KEY_MIME, mediaType);
        }
        return format;
    }
}
