package tv.twitch.android.player;

import android.media.MediaCodec;
import android.media.MediaCodecInfo;
import android.media.MediaCrypto;
import android.media.MediaFormat;
import android.os.Build;
import android.util.Log;
import android.util.Range;
import android.view.Surface;

import java.nio.ByteBuffer;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * Decodes input media samples for playback using {@link MediaCodec}.
 *
 * @author Nikhil Purushe
 */
@SuppressWarnings("unused") // called from native
class MediaCodecDecoder implements MediaDecoder, VideoRenderer.SurfaceChangeListener {

    private static final String MTK_AVC_DECODER = "OMX.MTK.VIDEO.DECODER.AVC";
    private static final String SEC_AVC_DECODER = "OMX.SEC.avc.dec";
    private static final Size SIZE_1080P = new Size(1920, 1080);
    private static final Size SIZE_1088P = new Size(1920, 1088);
    private static final Size SIZE_2160P = new Size(3840, 2160);
    private static final int NAL_UNIT_TYPE_SPS = 7;
    private static final long DEQUE_TIMEOUT_US = TimeUnit.MILLISECONDS.toMicros(5);

    private MediaFormat format;
    private MediaCodec codec;
    private String codecName;
    private final MediaRenderer renderer;
    private MediaCrypto mediaCrypto;
    private boolean rendererConfigured;
    private boolean hasSurface;
    private boolean supportsAdaptivePlayback;
    private boolean avcDiscardToSPS;
    private boolean configured;
    private boolean flushUnsupported;
    private boolean isVideo;
    private boolean useSoftwareDecoder;
    private boolean reconfigureRetry;
    private long reconfigureRetryTimeMillis;
    private long initialSampleTime;
    private MediaCodec.BufferInfo decodeInfo;
    private Set<Long> decodeOnlyBuffers;
    private Size maxDecodeSize;
    private int inputBufferIndex;
    private int queuedBufferCount;

    MediaCodecDecoder(MediaFormat format, MediaRenderer renderer) {
        this.renderer = renderer;
        this.format = format;
        this.decodeInfo = new MediaCodec.BufferInfo();
        this.decodeOnlyBuffers = new LinkedHashSet<>();
        this.inputBufferIndex = MediaCodec.INFO_TRY_AGAIN_LATER;
    }

    MediaCodecDecoder(MediaFormat format, SurfaceRenderer renderer) {
        this(format, (MediaRenderer)renderer);
        this.isVideo = true;
        this.maxDecodeSize = SIZE_1080P;
        renderer.setSurfaceChangeListener(this);
    }

    @Override
    public void configure(MediaFormat format) {
        this.format = format;
        // check if codec needs to be recreated in secure mode or crypto data changed
        MediaCrypto crypto = null;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            DrmSession drmSession = DrmSession.get(format);
            crypto = drmSession == null ? null : drmSession.getMediaCrypto(format);
        }
        if (mediaCrypto != crypto) {
            mediaCrypto = crypto;
            releaseCodec();
        }
        // initialize codec
        if (codec == null) {
            createCodec();
        }

        // if never configured configure once
        if (!configured) {
            reconfigure();
        }
        // adaptive video stream will reconfigure on next sps/pps sample
        // if not adaptive send EOS and then reconfigure
    }

    private void createCodec() {
        String mime = format.getString(MediaFormat.KEY_MIME);
        boolean requireSecureDecoder = mediaCrypto != null
                && mediaCrypto.requiresSecureDecoderComponent(mime);

        if (isVideo && useSoftwareDecoder && !requireSecureDecoder) {
            codec = MediaCodecFactory.createSoftwareCodec(mime);
        } else {
            codec = MediaCodecFactory.createCodec(mime, requireSecureDecoder);
        }

        if (codec == null) {
            throw new IllegalStateException("Failed to create codec instance " + mime);
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            codecName = codec.getName();
            // Work around for MediaTek AVC decoder

            if (MTK_AVC_DECODER.equalsIgnoreCase(codecName) &&
                    Build.VERSION.SDK_INT <= Build.VERSION_CODES.KITKAT) {
                avcDiscardToSPS = true;
            }
        } else {
            codecName = mime;
        }

        flushUnsupported = isFlushUnsupported();
        Log.i(Logging.TAG, "decode " + mime + " using " + codecName);
        checkCodecCapabilities(mime);
    }

    private void reconfigure() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            resetCodec();
        } else {
            configured = false;
        }

        // see https://developer.android.com/reference/android/media/MediaFormat.html
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT && supportsAdaptivePlayback && isVideo) {
            format.setInteger(MediaFormat.KEY_MAX_WIDTH, maxDecodeSize.width);
            format.setInteger(MediaFormat.KEY_MAX_HEIGHT, maxDecodeSize.height);
        }

        try {
            Surface surface = isVideo ? ((VideoRenderer) renderer).getSurface() : null;
            codec.configure(format, surface, mediaCrypto, 0);
            hasSurface = surface != null;
            Log.i(Logging.TAG, "configured " + codecName + " format " + format);
            codec.start();
            configured = true;
        } catch (Exception e) {
            Log.w(Logging.TAG, codecName + " configure failed", e);

            if (isVideo) {
                releaseCodec();

                if (!useSoftwareDecoder &&
                    Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP &&
                    e instanceof MediaCodec.CodecException) {
                    MediaCodec.CodecException ce = (MediaCodec.CodecException) e;
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                        useSoftwareDecoder = ce.getErrorCode() == MediaCodec.CodecException.ERROR_INSUFFICIENT_RESOURCE;
                    } else {
                        useSoftwareDecoder = true;
                    }
                }

                // reset reconfigure retry flag if the delta is higher than 1 minute.
                long currentTimeMillis = System.currentTimeMillis();
                if ((currentTimeMillis - reconfigureRetryTimeMillis) > 1000*60) {
                    reconfigureRetry = false;
                    reconfigureRetryTimeMillis = currentTimeMillis;
                }

                if (!reconfigureRetry) {
                    reconfigureRetry = true;
                    createCodec();
                    reconfigure();
                    return;
                }

                // [Temporary solution]
                // Added more debug info since we don't have those devices.
                throw new IllegalArgumentException("codecName:" + codecName +
                        " software:" + useSoftwareDecoder +
                        " surface: " + hasSurface +
                        " maxDecodeSize:" + maxDecodeSize.toString() +
                        " sap:" + supportsAdaptivePlayback +
                        " sps:" + avcDiscardToSPS +
                        " flush:" + flushUnsupported +
                        " format:" + format.toString() +
                        " e:" + e);

            }

            // if not recoverable rethrow the error
            throw e;
        }
    }

    private void checkCodecCapabilities(String mime) {
        if (isVideo) {
            maxDecodeSize = SIZE_1080P;
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            MediaCodecInfo.CodecCapabilities capabilities = null;
            try {
                capabilities = codec.getCodecInfo().getCapabilitiesForType(mime);
            } catch (IllegalArgumentException e) {
                // some 4.4 devices are not implementing getCapabilitiesForType correctly
                Log.e(Logging.TAG, "Failed to get codec capabilities", e);
            }
            if (capabilities != null) {
                if (isVideo) {
                    supportsAdaptivePlayback = capabilities.isFeatureSupported(
                            MediaCodecInfo.CodecCapabilities.FEATURE_AdaptivePlayback);
                }

                // on lollipop and above check for 4k support
                if (isVideo && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    MediaCodecInfo.VideoCapabilities video = capabilities.getVideoCapabilities();
                    if (video.getSupportedWidths().contains(SIZE_2160P.width)) {
                        Range<Integer> range = video.getSupportedHeightsFor(SIZE_2160P.width);
                        if (range.contains(SIZE_2160P.height)) {
                            maxDecodeSize = SIZE_2160P;
                        }
                    }
                }
            }
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            if (isVideo && MediaCodecFactory.limitMaxDecodeSize(codec.getName())) {
                maxDecodeSize = SIZE_1088P;
            }
        }
    }

    @Override
    public boolean hasInput() {
        if (isVideo) {
            // if the video surface has changed return true to reconfigure the codec
            if (!configured) {
                return true;
            }
        }
        if (configured && codec != null) {
            try {
                long timeout = queuedBufferCount > 0 ? DEQUE_TIMEOUT_US : 0;
                inputBufferIndex = codec.dequeueInputBuffer(timeout);
            } catch (IllegalStateException e) {
                Log.w(Logging.TAG, codecName + " MediaCodecDecoder dequeueInputBuffer failed", e);
                // force a reconfiguration
                resetCodec();
            }
            return inputBufferIndex >= 0;
        }
        return false;
    }

    @Override
    public void decode(MediaSample sample) {
        if (initialSampleTime <= 0) {
            initialSampleTime = sample.decodeTimeUs;
        }

        // in the case of the surface changing we drop video samples until a keyframe
        if (isVideo) {
            VideoRenderer videoRenderer = (VideoRenderer) renderer;
            if (!configured) {
                if (!sample.isSyncSample || videoRenderer.getSurface() == null) {
                    configured = false;
                    return;
                } else {
                    Log.w(Logging.TAG, "MediaCodecDecoder switching surface");
                    releaseCodec();
                    createCodec();
                    reconfigure();
                    videoRenderer.configure(format); // this clears the isSurfaceChanged flag
                    try {
                        inputBufferIndex = codec.dequeueInputBuffer(-1);
                    } catch (Exception e) {
                        Log.w(Logging.TAG, codecName + " MediaCodecDecoder dequeueInputBuffer failed", e);
                        // force a reconfiguration
                        resetCodec();
                    }
                }
            }
        }

        // inputBufferIndex should be set from hasInput()
        if (inputBufferIndex < 0) {
            throw new MediaException(MediaResult.ErrorInvalidParameter, "invalid buffer index");
        }

        try {
            queueInput(inputBufferIndex, sample);
        } catch (Exception e) {
            Log.w(Logging.TAG, codecName + " MediaCodecDecoder queueInput failed", e);
            // force a reconfiguration
            resetCodec();
        }

        inputBufferIndex = MediaCodec.INFO_TRY_AGAIN_LATER;
    }

    private void queueInput(int index, MediaSample sample) {
        ByteBuffer buffer = getInputBuffer(index);
        if (avcDiscardToSPS) {
            avcDiscardToSPS(sample);
        }
        buffer.put(sample.buffer);
        long presentationTimeUs = sample.presentationTimeUs - initialSampleTime;
        int flags = 0;
        if (sample.isSyncSample) {
            flags |= MediaCodec.BUFFER_FLAG_SYNC_FRAME;
        }
        if (sample.cryptoInfo == null) {
            codec.queueInputBuffer(index, 0, sample.size, presentationTimeUs, flags);
        } else {
            codec.queueSecureInputBuffer(index, 0, sample.cryptoInfo, presentationTimeUs, flags);
        }
        queuedBufferCount++;
        if (sample.isDecodeOnly) {
            decodeOnlyBuffers.add(sample.presentationTimeUs);
        }
    }

    private boolean dequeOutput(long timeoutUs) {
        int index = codec.dequeueOutputBuffer(decodeInfo, timeoutUs);
        if (index >= 0) {
            queuedBufferCount--;
            if ((decodeInfo.flags & MediaCodec.BUFFER_FLAG_END_OF_STREAM) != 0) {
                Log.d(Logging.TAG, codecName + " BUFFER_FLAG_END_OF_STREAM");
                resetCodec();
                return false;
            }

            long presentationTime = initialSampleTime + decodeInfo.presentationTimeUs;
            boolean render = !decodeOnlyBuffers.contains(presentationTime);

            if (isVideo) {
                // if video render directly to the surface
                if (render && renderer != null) {
                    if (!rendererConfigured) {
                        try {
                            MediaFormat outputFormat = getOutputFormat(index);
                            configureRenderer(outputFormat);
                        } catch (IllegalStateException e) {
                            // happens on LG G Flex and possibly other 4.x devices
                            Log.d(Logging.TAG, "Failed to get output format", e);
                            rendererConfigured = true;
                            configureRenderer(format);
                        }
                    }
                    VideoRenderer videoRenderer = (VideoRenderer) renderer;
                    videoRenderer.render(codec, index, presentationTime);
                    return inputBufferIndex < 0;
                } else {
                    codec.releaseOutputBuffer(index, false);
                }
            } else {
                ByteBuffer output = getOutputBuffer(index);
                if (render && renderer != null && rendererConfigured) {
                    renderer.render(output, decodeInfo.size, presentationTime);
                }
                codec.releaseOutputBuffer(index, false);
            }
        } else if (index == MediaCodec.INFO_OUTPUT_FORMAT_CHANGED) {
            Log.d(Logging.TAG, codecName + " INFO_OUTPUT_FORMAT_CHANGED");
            configureRenderer(codec.getOutputFormat());
            return true;
        } else if (index == MediaCodec.INFO_OUTPUT_BUFFERS_CHANGED) {
            Log.d(Logging.TAG, codecName + " INFO_OUTPUT_BUFFERS_CHANGED");
            return true;
        } // MediaCodec.INFO_TRY_AGAIN_LATER or unknown value
        return false;
    }

    @Override
    public void flush() {
        if (codec == null || !configured) {
            return;
        }
        if (flushUnsupported) {
            releaseCodec(); // codec will have to be recreated
        } else if (!supportsAdaptivePlayback) {
            // if the queued buffer count is zero don't wait for any output
            if (queuedBufferCount == 0) {
                Log.w(Logging.TAG, codecName + " no buffers queued on flush");
                resetCodec();
                return;
            }
            // for non adaptive first queue the EOS buffer
            int index = -1;
            while (index < 0) {
                index = codec.dequeueInputBuffer(DEQUE_TIMEOUT_US);
                if (index < 0) {
                    Log.w(Logging.TAG, codecName + " wait to queue EOS");
                    dequeOutput(0);
                } else {
                    Log.i(Logging.TAG, codecName + " queue BUFFER_FLAG_END_OF_STREAM");
                    codec.queueInputBuffer(index, 0, 0, 0, MediaCodec.BUFFER_FLAG_END_OF_STREAM);
                }
            }
            // drain the decoder until the EOS buffer is reached
            while (configured) {
                dequeOutput(0);
            }
        }
    }

    @Override
    public void release() {
        releaseCodec();
    }

    @Override
    public void reset() {
        initialSampleTime = 0;
        resetDecodeInfo();
        if (codec != null && configured) {
            if (!supportsAdaptivePlayback || flushUnsupported) {
                resetCodec();
            } else {
                try {
                    // if supports adaptive playback/flush just flush the codec and reuse it
                    codec.flush();
                } catch (IllegalStateException e) {
                    e.printStackTrace();
                    resetCodec();
                }
            }
        }
    }

    @Override
    public boolean hasOutput() {
        try {
            return configured && dequeOutput(0);
        } catch (IllegalStateException e) {
            e.printStackTrace();
            Log.w(Logging.TAG, codecName + " MediaCodecDecoder dequeOutput failed", e);
            return false;
        }
    }

    @Override
    public long getOutputTime() {
        return initialSampleTime + decodeInfo.presentationTimeUs;
    }

    @Override
    public void onSurfaceChange(Surface surface) {
        if (configured && codec != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M && hasSurface && surface != null) {
                codec.setOutputSurface(surface);
            } else {
                releaseCodec();
            }
        }
    }

    private void configureRenderer(MediaFormat format) {
        if (renderer != null) {
            renderer.configure(format);
            rendererConfigured = true;
        }
    }

    private void resetCodec() {
        if (codec != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                codec.reset();
            } else {
                releaseCodec();
                createCodec();
            }
        }
        configured = false;
        hasSurface = false;
    }

    private void releaseCodec() {
        resetDecodeInfo();
        if (codec != null) {
            try {
                if (configured) {
                    codec.stop();
                }
            } catch (Exception e) {
                Log.w(Logging.TAG, "Codec stop() failed", e);
            } finally {
                try {
                    codec.release();
                } catch (Exception e) {
                    Log.w(Logging.TAG, "Codec release() failed", e);
                } finally {
                    codec = null;
                }
            }
        }
        configured = false;
        hasSurface = false;
    }

    private void resetDecodeInfo() {
        decodeOnlyBuffers.clear();
        decodeInfo = new MediaCodec.BufferInfo();
        queuedBufferCount = 0;
    }

    private MediaFormat getOutputFormat(int index) {
        MediaFormat format;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            format = codec.getOutputFormat(index);
        } else {
            format = codec.getOutputFormat();
        }
        return format;
    }

    private ByteBuffer getInputBuffer(int index) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            return codec.getInputBuffer(index);
        } else {
            return codec.getInputBuffers()[index];
        }
    }

    private ByteBuffer getOutputBuffer(int index) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            return codec.getOutputBuffer(index);
        } else {
            return codec.getOutputBuffers()[index];
        }
    }

    private boolean isFlushUnsupported() {
        return Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2 ||
               (Build.VERSION.SDK_INT == Build.VERSION_CODES.JELLY_BEAN_MR2 &&
                       (codecName != null && codecName.startsWith(SEC_AVC_DECODER)));
    }

    private static void avcDiscardToSPS(MediaSample sample) {
        ByteBuffer buffer = sample.buffer;
        int precedingZeroCount = 0;

        for (int i = 0; i < sample.size; i++) {
            int value = buffer.get(i) & 0xFF;
            boolean hasStartCode = precedingZeroCount == 3 && value == 1;
            if (hasStartCode) {
                if (i + 1 >= sample.size) {
                    return;
                }
                int unitType = (buffer.get(i + 1) & 0x1F);
                if (unitType == NAL_UNIT_TYPE_SPS) {
                    ByteBuffer copy = buffer.duplicate();
                    copy.position(i - precedingZeroCount);
                    buffer.rewind();
                    buffer.limit(copy.remaining());
                    buffer.put(copy);
                    buffer.position(0);
                    return;
                }
            } else if (value == 0) {
                precedingZeroCount++;
            } else {
                precedingZeroCount = 0;
            }
        }
    }
}
