package tv.twitch.android.player;

import android.media.MediaCodec;
import android.media.MediaCodecInfo;
import android.media.MediaCodecList;
import android.util.Log;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

class MediaCodecFactory {

    private MediaCodecFactory() {}

    static MediaCodec createCodec(String mime, boolean secure) {
        MediaCodec codec = null;
        if (!secure) {
            try {
                codec = MediaCodec.createDecoderByType(mime);
            } catch (IOException e) {
                Log.w(Logging.TAG, "Failed to create codec " + mime, e);
            }
        }
        // if secure codec required or failed to create, create it by lookup
        if (codec == null) {
            String codecName = findDecoder(mime, secure);
            codec = createCodecByName(codecName);
        }

        if (codec == null && secure) {
            // https://developer.android.com/reference/android/media/MediaCodec#creation
            // see section Creating secure decoders
            String codecName = findDecoder(mime, false);
            if (codecName != null) {
                codec = createCodecByName(codecName + ".secure");
            }
        }
        return codec;
    }

    private static MediaCodec createCodecByName(String name) {
        if (name != null) {
            try {
                return MediaCodec.createByCodecName(name);
            } catch (IOException e) {
                Log.w(Logging.TAG, "Failed to create codec " + name, e);
            }
        }
        return null;
    }

    static final String OMX_SOFTWARE_AVC_DECODER = "OMX.google.h264.decoder";
    static final String OMX_SOFTWARE_VP9_DECODER = "OMX.google.vp9.decoder";

    static MediaCodec createSoftwareCodec(String mediaType) {
        switch (mediaType) {
            case MediaType.VIDEO_AVC:
                return createCodecByName(OMX_SOFTWARE_AVC_DECODER);
            case MediaType.VIDEO_VP9:
                return createCodecByName(OMX_SOFTWARE_VP9_DECODER);
            default:
                return null;
        }
    }

    @SuppressWarnings("deprecation")
    private static String findDecoder(String mime, boolean secure) {
        List<MediaCodecInfo> supported = findSupportedCodecs(mime, secure);
        if (!supported.isEmpty()) {
            MediaCodecInfo info = supported.get(0);
            return info.getName();
        }
        return null;
    }

    @SuppressWarnings("deprecation")
    static List<MediaCodecInfo> findSupportedCodecs(String mime, boolean secure) {
        List<MediaCodecInfo> supported = new ArrayList<>();
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
            int kind = secure ? MediaCodecList.ALL_CODECS : MediaCodecList.REGULAR_CODECS;
            MediaCodecList list = new MediaCodecList(kind);

            for (MediaCodecInfo info : list.getCodecInfos()) {
                if (info.isEncoder() || !supportsType(info, mime)) {
                    continue;
                }
                boolean isSupported = !secure;
                if (secure) {
                    MediaCodecInfo.CodecCapabilities capabilities = info.getCapabilitiesForType(mime);
                    if (capabilities.isFeatureSupported(
                            MediaCodecInfo.CodecCapabilities.FEATURE_SecurePlayback)) {
                        isSupported = true;
                    }
                }
                if (isSupported) {
                    supported.add(info);
                }
            }
        } else {
            for (int i = 0; i < MediaCodecList.getCodecCount(); i++) {
                MediaCodecInfo info = MediaCodecList.getCodecInfoAt(i);
                if (info.isEncoder() || !supportsType(info, mime)) {
                    continue;
                }
                boolean isSupported = !secure;
                if (secure && info.getName().toLowerCase().contains("sec")) {
                    isSupported = true;
                }
                if (isSupported) {
                    supported.add(info);
                }
            }
        }

        return supported;
    }

    private static boolean supportsType(MediaCodecInfo info, String mime) {
        String[] types = info.getSupportedTypes();
        for (String type : types) {
            if (type.equalsIgnoreCase(mime)) {
                return true;
            }
        }
        return false;
    }

    static boolean limitMaxDecodeSize(String codecName) {
        // Performance, crash, freeze or corrupt screen issues.
        // This is a workaround where we need to set max video size at 1920x1088.
        return "OMX.amlogic.avc.decoder.awesome".equals(codecName);
    }
}
