package tv.twitch.android.player;

import android.content.Context;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.Surface;

import java.nio.ByteBuffer;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Native player implementation using the player-core library.
 *
 * @author Nikhil Purushe
 */
public class MediaPlayer implements Player, VideoRenderer.VideoSizeListener {

    private Platform platform;
    private long player;
    private final Handler handler;
    private final Set<Listener> listeners;
    private final AtomicBoolean closed;
    private Statistics statistics;

    public MediaPlayer(Context context) {
        this.platform = new Platform(context, this);
        this.player = init(platform, context.getCacheDir().toString());
        this.handler = new Handler(Looper.getMainLooper());
        this.closed = new AtomicBoolean();
        this.listeners = new CopyOnWriteArraySet<>();
        this.statistics = new Statistics();
    }

    static {
        try {
            System.loadLibrary(Platform.LIBRARY_NAME);
        } catch (UnsatisfiedLinkError e) {
            Log.e(Logging.TAG, "Failed to load " + Platform.LIBRARY_NAME, e);
        }
        initialize();
    }

    // native methods
    private static native void initialize();
    private native long init(Platform platform, String tempPath);
    private native void load(long player, String path, String mediaType);
    private native boolean isSeekable(long player);
    private native void seekTo(long player, long position);
    private native void play(long player);
    private native void pause(long player);
    private native void release(long player);
    private native boolean isMuted(long player);
    private native int getState(long player);
    private native long getPosition(long player);
    private native long getAverageBitrate(long player);
    private native long getBandwidthEstimate(long player);
    private native long getBufferedPosition(long player);
    private native long getLiveLatency(long player);
    private native long getDuration(long player);
    private native HashSet<Quality> getQualities(long player);
    private native Quality getQuality(long player);
    private native void setQuality(long player, String name, boolean adaptive);
    private native boolean getAutoSwitchQuality(long player);
    private native void setAutoSwitchQuality(long player, boolean enable);
    private native void setAutoMaxVideoSize(long player, int width, int height);
    private native void setAutoInitialBitrate(long player, int bitrate);
    private native void setAutoMaxBitrate(long player, int bitrate);
    private native void setMinBuffer(long player, long duration);
    private native void setMaxBuffer(long player, long duration);
    private native void setLiveLowLatencyEnabled(long player, boolean enabled);
    private native void setLooping(long player, boolean loop);
    private native void setMuted(long player, boolean muted);
    private native void setSurface(long player, Surface surface);
    private native void setPlaybackRate(long player, float rate);
    private native void setVolume(long player, float volume);
    private native void setClientId(long player, String id);
    private native void setDeviceId(long player, String id);
    private native void getStatistics(long player, Statistics statistics);

    @Override
    protected void finalize() throws Throwable {
        try {
            if (player != 0) {
                Log.e(Logging.TAG, "Player not released, releasing from finalizer");
            }
            release();
        } finally {
            super.finalize();
        }
    }

    @SuppressWarnings("unused") // called from native
    void handleError(int source, int result, int code, String message) {
        final MediaException exception = platform.createException(source, result, code, message);
        handler.post(new Runnable() {
            @Override
            public void run() {
                for (Listener listener : listeners) {
                    listener.onError(exception);
                }
            }
        });
    }

    @SuppressWarnings("unused") // called from native
    void handleQualityChange(final Quality quality) {
        handler.post(new Runnable() {
            @Override
            public void run() {
                for (Listener listener : listeners) {
                    listener.onQualityChanged(quality);
                }
            }
        });
    }

    @SuppressWarnings("unused") // called from native
    void handleRebuffering() {
        handler.post(new Runnable() {
            @Override
            public void run() {
                for (Listener listener : listeners) {
                    listener.onRebuffering();
                }
            }
        });
    }

    @SuppressWarnings("unused") // called from native
    void handleStateChange(final int value) {
        handler.post(new Runnable() {
            @Override
            public void run() {
                State state = mapState(value);
                for (Listener listener : listeners) {
                    listener.onStateChanged(state);
                }
            }
        });
    }

    @SuppressWarnings("unused") // called from native
    void handleAnalyticsEvent(final String name, final String properties) {
        handler.post(new Runnable() {
            @Override
            public void run() {
                for (Listener listener : listeners) {
                    listener.onAnalyticsEvent(name, properties);
                }
            }
        });
    }

    @SuppressWarnings("unused") // called from native
    void handleMetadata(final String type, final ByteBuffer data) {
        // buffer is only valid for this function invocation so copying is required
        final ByteBuffer copy = ByteBuffer.allocate(data.capacity());
        copy.put(data);
        copy.flip();
        handler.post(new Runnable() {
            @Override
            public void run() {
                for (Listener listener : listeners) {
                    listener.onMetadata(type, copy);
                }
            }
        });
    }

    private State mapState(int state) {
        switch (state) {
            default:
            case 0:
                return State.IDLE;
            case 1:
                return State.READY;
            case 2:
                return State.BUFFERING;
            case 3:
                return State.PLAYING;
            case 4:
                return State.ENDED;
        }
    }

    @Override
    public void release() {
        if (closed.compareAndSet(false, true)) {
            handler.removeCallbacksAndMessages(null);
            release(player);
            player = 0;
        }
    }

    @Override
    public void addListener(Listener listener) {
        if (listener != null) {
            listeners.add(listener);
        }
    }

    @Override
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }

    @Override
    public int getAudioSessionId() {
        return platform.getAudioSessionId();
    }

    @Override
    public State getState() {
        return mapState(getState(player));
    }

    @Override
    public long getPosition() {
        return getPosition(player);
    }

    @Override
    public long getDuration() {
        return getDuration(player);
    }

    @Override
    public long getAverageBitrate() {
        return getAverageBitrate(player);
    }

    @Override
    public long getBandwidthEstimate() {
        return getBandwidthEstimate(player);
    }

    @Override
    public long getBufferedPosition() {
        return getBufferedPosition(player);
    }

    @Override
    public long getLiveLatency() {
        return getLiveLatency(player);
    }

    @Override
    public String getVersion() {
        return BuildConfig.VERSION_NAME;
    }

    @Override
    public boolean isMuted() {
        return isMuted(player);
    }

    @Override
    public void setLooping(boolean loop) {
        setLooping(player, loop);
    }

    @Override
    public void setMuted(boolean muted) {
        setMuted(player, muted);
    }

    @Override
    public void setSurface(Surface surface) {
        setSurface(player, surface);
    }

    @Override
    public void setPlaybackRate(float rate) {
        setPlaybackRate(player, rate);
    }

    @Override
    public void setVolume(float volume) {
        setVolume(player, volume);
    }

    @Override
    public void load(Uri uri) {
        load(player, uri.toString(), "");
    }

    @Override
    public void load(Uri uri, String mediaType) {
        load(player, uri.toString(), mediaType);
    }

    @Override
    public boolean isSeekable() {
        return isSeekable(player);
    }

    @Override
    public void seekTo(long position) {
        seekTo(player, position);
    }

    @Override
    public void play() {
        play(player);
    }

    @Override
    public void pause() {
        pause(player);
    }

    @Override
    public Set<Quality> getQualities() {
        return getQualities(player);
    }

    @Override
    public Quality getQuality() {
        return getQuality(player);
    }

    @Override
    public void setQuality(Quality quality) {
        setQuality(player, quality.getName(), false);
    }

    @Override
    public void setQuality(Quality quality, boolean adaptive) {
        if (quality.getName() == null) {
            throw new IllegalArgumentException();
        }
        setQuality(player, quality.getName(), adaptive);
    }

    @Override
    public boolean getAutoSwitchQuality() {
        return getAutoSwitchQuality(player);
    }

    @Override
    public void setAutoSwitchQuality(boolean enable) {
        setAutoSwitchQuality(player, enable);
    }

    @Override
    public void setAutoInitialBitrate(int bitrate) {
        setAutoInitialBitrate(player, bitrate);
    }

    @Override
    public void setAutoMaxBitrate(int bitrate) {
        setAutoMaxBitrate(player, bitrate);
    }

    @Override
    public void setAutoMaxVideoSize(int width, int height) {
        setAutoMaxVideoSize(player, width, height);
    }

    @Override
    public void setLiveLowLatencyEnabled(boolean enabled) {
        setLiveLowLatencyEnabled(player, enabled);
    }

    @Override
    public void setMinBuffer(int durationMs) {
        setMinBuffer(player, durationMs);
    }

    @Override
    public void setMaxBuffer(int durationMs) {
        setMaxBuffer(player, durationMs);
    }

    @Override
    public void setClientId(String id) {
        setClientId(player, id);
    }

    @Override
    public void setDeviceId(String id) {
        setDeviceId(player, id);
    }

    @Override
    public Statistics getStatistics() {
        getStatistics(player, statistics);
        return statistics;
    }

    @Override
    public void onSizeChange(final int width, final int height) {
        handler.post(new Runnable() {
            @Override
            public void run() {
                for (Listener listener : listeners) {
                    listener.onVideoSizeChanged(width, height);
                }
            }
        });
    }
}
