package tv.twitch.android.player;

import android.content.Context;
import android.media.MediaFormat;
import android.os.Process;
import android.util.Log;

import java.lang.ref.WeakReference;
import java.nio.ByteBuffer;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;

public class Platform {

    public static final String LIBRARY_NAME = BuildConfig.PLAYERCORE_LIB;

    private static ErrorListener errorListener;
    private final Context context;
    private AudioTrackRenderer audioRenderer;
    private SurfaceRenderer surfaceRenderer;
    private ConcurrentLinkedQueue<Throwable> pendingExceptions;
    private WeakReference<VideoRenderer.VideoSizeListener> videoSizeListener;

    public Platform(Context context) {
        this(context, null);
    }

    public Platform(Context context, VideoRenderer.VideoSizeListener videoSizeListener) {
        this.context = context;
        this.pendingExceptions = new ConcurrentLinkedQueue<>();
        if (videoSizeListener != null) {
            this.videoSizeListener = new WeakReference<>(videoSizeListener);
        }
    }

    public interface ErrorListener {
        void onError(Throwable t);
    }

    // sets a global error listener for the library
    public static void setErrorListener(ErrorListener listener) {
        errorListener = listener;
    }

    @SuppressWarnings("unused") // called from native
    MediaDecoder createDecoder(MediaFormat format) {
        MediaDecoder decoder = null;
        String mime = format.getString(MediaFormat.KEY_MIME);
        if (mime.startsWith(MediaType.TYPE_VIDEO)) {
            // only create the surface renderer once
            if (surfaceRenderer == null) {
                VideoRenderer.VideoSizeListener listener = videoSizeListener != null ? videoSizeListener.get() : null;
                surfaceRenderer = new SurfaceRenderer(context, listener);
            }
            return new MediaCodecDecoder(format, surfaceRenderer);
        } else if (mime.startsWith(MediaType.TYPE_AUDIO)) {
            if (audioRenderer != null) {
                audioRenderer.release();
            }
            audioRenderer = new AudioTrackRenderer();
            decoder = new MediaCodecDecoder(format, audioRenderer);
        }
        return decoder;
    }

    @SuppressWarnings("unused") // called from native
    synchronized MediaRenderer createRenderer(MediaFormat format) {
        String mime = format.getString(MediaFormat.KEY_MIME);
        if (mime.startsWith(MediaType.TYPE_AUDIO)) {
            return audioRenderer;
        } else if (mime.startsWith(MediaType.TYPE_VIDEO)) {
            return surfaceRenderer;
        }
        return null;
    }

    MediaException createException(int source, int result, int code, String message) {
        ErrorSource errorSource = ErrorSource.fromInt(source);
        MediaResult mediaResult = MediaResult.fromInt(result);
        final MediaException exception;
        Throwable cause = pendingExceptions.poll();
        if (cause != null) {
            exception = new MediaException(cause, errorSource, mediaResult, code, message);
            if (pendingExceptions.size() > 1) {
                Log.w(Logging.TAG, "Errored with multiple exceptions");
                for (Throwable t : pendingExceptions) {
                    Log.w(Logging.TAG, t);
                }
            }
            pendingExceptions.clear();
        } else {
            exception = new MediaException(errorSource, mediaResult, code, message);
        }
        return exception;
    }

    int getAudioSessionId() {
        return audioRenderer == null ? -1 : audioRenderer.getAudioSessionId();
    }

    @SuppressWarnings("unused") // called from native
    static ByteBuffer[] getSupportedProtectionSystemUUIDs() {
        ProtectionSystem[] schemes = ProtectionSystem.getSupported().toArray(new ProtectionSystem[0]);
        ByteBuffer[] result = new ByteBuffer[schemes.length];
        for (int i = 0; i < schemes.length; i++) {
            ByteBuffer buffer = ByteBuffer.allocateDirect(16);
            UUID id = schemes[i].getUUID();
            buffer.putLong(id.getMostSignificantBits());
            buffer.putLong(id.getLeastSignificantBits());
            result[i] = buffer;
        }
        return result;
    }

    @SuppressWarnings("unused") // called from native
    VideoCapabilities getVideoDecoderCapabilities(String mediaType) {
        return VideoCapabilities.getCapabilities(context, mediaType);
    }

    @SuppressWarnings("unused") // called from native
    public void handleDecoderException(Throwable t) {
        if (t != null) {
            Log.w(Logging.TAG, t);
            pendingExceptions.add(t);
            if (errorListener != null) {
                errorListener.onError(t);
            }
        }
    }

    @SuppressWarnings("unused") // called from native
    public void handleRendererException(Throwable t) {
        if (t != null) {
            Log.w(Logging.TAG, t);
            pendingExceptions.add(t);
            if (errorListener != null) {
                errorListener.onError(t);
            }
        }
    }

    @SuppressWarnings("unused") // called from native
    public void onThreadCreated(String name) {
        // set all threads to foreground for performance reasons
        try {
            android.os.Process.setThreadPriority(Process.THREAD_PRIORITY_FOREGROUND);
        } catch (Exception e) {
            Log.w(Logging.TAG, e);
        }
        // set thread name
        if (name != null && !name.isEmpty()) {
            Thread.currentThread().setName(name);
        }
    }
}
