package tv.twitch.android.player;

import android.net.Uri;
import android.view.Surface;

import java.nio.ByteBuffer;
import java.util.Set;

/**
 * Interface for controlling a Player instance. API users instantiate a player implementation
 * and issue commands to it. The commands are generally asynchronous in nature, the player
 * communicates is state back to the user via the {@link Listener} interface. The two primary
 * callbacks are for {@link State} changes via {@link Listener#onStateChanged(Player.State)} and for
 * errors {@link Listener#onError(MediaException)}.
 */
public interface Player extends Releasable {

    /**
     * Represents the current state of a player instance.
     */
    enum State {
        /**
         * Player is buffering for playback.
         */
        BUFFERING,
        /**
         * Player is playing.
         */
        PLAYING,
        /**
         * Player is ready.
         */
        READY,
        /**
         * Player is idle. This is the initial state.
         */
        IDLE,
        /**
         * Player is finished playing the stream.
         */
        ENDED,
    }

    /**
     * Player listener interface
     */
    interface Listener {
        /**
         * Called with a video analytics tracking event
         * @param name name of the event
         * @param properties properties for the event (json)
         */
        void onAnalyticsEvent(String name, String properties);

        /**
         * Indicates a state change in the player
         *
         * @param state current player state
         */
        void onStateChanged(State state);

        /**
         * Indicates an error in the player. The player will move into {@link State#IDLE} on an error.
         *
         * @param exception exception info
         */
        void onError(MediaException exception);

        /**
         * Indicates the player is buffering from a previous playing state.
         */
        void onRebuffering();

        /**
         * Called after the video dimensions have changed.
         *
         * @param width video width
         * @param height video height
         */
        void onVideoSizeChanged(int width, int height);

        /**
         * Indicates a change in the playing quality
         *
         * @param quality current quality
         */
        void onQualityChanged(Quality quality);

        /**
         * Called on a metadata event e.g. ID3
         * @param mediaType mime type e.g. 'text/id3'
         * @param data metadata content format depends on the type of metadata.
         */
        void onMetadata(String mediaType, ByteBuffer data);
    }

    /**
     * @return Audio session id currently used by the audio track.
     */
    int getAudioSessionId();

    /**
     * @return the current state of the player
     */
    State getState();

    /**
     * @return the current playback position in milliseconds of the player, if the player is in
     * the {@link State#PLAYING} state, otherwise returns the last play position or -1 if not
     * started yet.
     */
    long getPosition();

    /**
     * @return the duration of the currently loaded stream in milliseconds, -1 if no stream is
     * loaded yet or the stream length is infinite or unknown.
     */
    long getDuration();

    /**
     * @return the current playback position in milliseconds of the player or -1 if no data is
     * buffered.
     */
    long getBufferedPosition();

    /**
     * @return version of the player
     */
    String getVersion();

    /**
     * Adds a playback listener that receives information about this player instance.
     *
     * @param listener to add
     */
    void addListener(Listener listener);

    /**
     * Removes a playback state listener.
     *
     * @param listener to remove
     */
    void removeListener(Listener listener);

    /**
     * Enables the player to loop the source video if supported.
     *
     * @param loop true to loop video playback, false otherwise
     */
    void setLooping(boolean loop);

    /**
     * @return true if the player is muted, false otherwise
     */
    boolean isMuted();

    /**
     * Mutes or unmutes the audio of the player
     *
     * @param muted true to mute the player false to unmute the player
     */
    void setMuted(boolean muted);

    /**
     * Set's the {@link Surface to use for} rendering video, this method can be called at any time
     * on the player.
     *
     * @param surface surface to use for rendering. this value can be null to  stop displaying video
     */
    void setSurface(Surface surface);

    /**
     * Sets the playback rate for the player.
     *
     * @param rate rate to set
     */
    void setPlaybackRate(float rate);

    /**
     * Sets the output gain value for the audio track.
     *
     * @param volume volume for the audio track between 0.0f and 1.0f
     */
    void setVolume(float volume);

    /**
     * Loads the given stream. On success the player state will change from it's current state
     * to Ready. On failure invokes the onError listener.
     *
     * @param uri of the stream to load (e.g. a url or file)
     */
    void load(Uri uri);

    /**
     * Loads the given stream. On success the player state will change from it's current state
     * to Ready. On failure invokes the onError listener.
     *
     * @param uri of the stream to load (e.g. a url or file)
     * @param mediaType Media type of the content if known, e.g. "video/mp4" or "application/x-mpegURL"
     */
    void load(Uri uri, String mediaType);

    /**
     * @return true if seeking is supported in the currently loaded stream, false otherwise.
     */
    boolean isSeekable();

    /**
     * Seeks to a specific position in the stream at the given time in milliseconds.
     *
     * @param position to seek to in milliseconds
     */
    void seekTo(long position);

    /**
     * Starts or resumes playback of the current stream or fails if no stream is loaded.
     * On success depending on the type of stream the player state will change to
     * {@link State#BUFFERING} and then {@link State#PLAYING} or just {@link State#PLAYING}.
     * On failure invokes the onError callback.
     */
    void play();

    /**
     * Pauses playback of the current stream or fails if no stream is loaded. On failure invokes
     * the onError callback.
     *
     * Resuming the stream with play may result in the position being different depending on the
     * type of source being played.
     */
    void pause();

    /**
     * @return set of {@link Quality} which represent different qualities
     * available in the stream being played.
     */
    Set<Quality> getQualities();

    /**
     * @return the current {@link Quality} of the source
     */
    Quality getQuality();

    /**
     * Sets the current quality of the stream, if auto switch quality it is disabled.
     * @param quality to set
     */
    void setQuality(Quality quality);

    /**
     * Sets the current quality of the stream, if auto switch quality it is disabled.
     * @param quality to set
     * @param adaptive true for an adaptive quality change (quality changes at the end
     * of the buffer, false otherwise buffer cleared immediately)
     */
    void setQuality(Quality quality, boolean adaptive);

    /**
     * @return true if quality auto switching is enabled, false otherwise
     */
    boolean getAutoSwitchQuality();

    /**
     * @param enable true to enable auto quality selection, false to disable
     */
    void setAutoSwitchQuality(boolean enable);

    /**
     * Sets the initial starting bitrate when using auto quality switching. This determines the
     * initial quality an adaptive stream will playback at when no bandwidth estimate value has
     * been determined.
     * @param bitrate initial bitrate to use
     */
    void setAutoInitialBitrate(int bitrate);

    /**
     * Sets the max bitrate to use when using auto quality switching. Allows controlling resource
     * usage in the case of multiple players.
     * @param bitrate bitrate to use
     */
    void setAutoMaxBitrate(int bitrate);

    /**
     * Sets the max video display size of the player, which prevents switching to qualities above
     * this resolution when autoSwitchQuality is true.
     * @param width display width
     * @param height display height
     */
    void setAutoMaxVideoSize(int width, int height);

    /**
     * Enables low latency playback for streams which support it. Defaults to true, changing the
     * value during playback will restart the stream.
     *
     * @param enabled true to enable, false to disable
     */
    void setLiveLowLatencyEnabled(boolean enabled);

    /**
     * Sets the minimum buffer size needed to start playback.
     * @param durationMs of buffer to set in milliseconds
     */
    void setMinBuffer(int durationMs);

    /**
     * Sets the target buffer size to buffer ahead of the playhead. Note in live streams this
     * buffer size may not be met.
     * @param durationMs of buffer to set in milliseconds
     */
    void setMaxBuffer(int durationMs);

    /**
     * Sets the API client id used for internal requests made by the player
     * @param id to set
     */
    void setClientId(String id);

    /**
     * Sets the device id for this player instance used for API requests
     * @param id to set
     */
    void setDeviceId(String id);

    /**
     * @return average source media bitrate of the stream
     */
    long getAverageBitrate();

    /**
     * @return for network streams the current ABR bandwidth estimate in bps or 0 if unknown.
     */
    long getBandwidthEstimate();

    /**
     * @return For a live stream the latency to the source in milliseconds.
     */
    long getLiveLatency();

    /**
     * @return Playback {@link Statistics} for the currently playing stream.
     */
    Statistics getStatistics();
}
