package tv.twitch.android.player;

import android.content.Context;
import android.media.MediaCodec;
import android.media.MediaFormat;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.Choreographer;
import android.view.Display;
import android.view.Surface;
import android.view.WindowManager;

import java.lang.ref.WeakReference;
import java.nio.ByteBuffer;
import java.util.concurrent.TimeUnit;

class SurfaceRenderer implements MediaRenderer, VideoRenderer, Choreographer.FrameCallback {

    private static final String FORMAT_KEY_CROP_LEFT = "crop-left";
    private static final String FORMAT_KEY_CROP_RIGHT = "crop-right";
    private static final String FORMAT_KEY_CROP_TOP = "crop-top";
    private static final String FORMAT_KEY_CROP_BOTTOM = "crop-bottom";

    private Surface surface;
    private final Object surfaceLock;
    private long renderPresentationTime = -1;
    private WeakReference<VideoSizeListener> sizeListener;
    private Size size;
    private long mediaTime;
    private long lastMediaTime = -1;
    private long startTime;
    private long syncInterval;
    private int renderedFrames;
    private int droppedFrames;
    private boolean sizeChanged;
    private float playbackRate = 1.0f;
    private long frameTimeNanos;
    private Handler handler;
    private Choreographer choreographer;
    private SurfaceChangeListener surfaceChangeListener;

    SurfaceRenderer(Context context, VideoSizeListener listener) {
        if (context == null) {
            throw new IllegalArgumentException();
        }
        this.surfaceLock = new Object();
        this.size = new Size(0, 0);
        this.sizeListener = new WeakReference<>(listener);
        // https://developer.android.com/reference/android/media/MediaCodec.html#releaseOutputBuffer(int, long)
        this.syncInterval = TimeUnit.MILLISECONDS.toMicros(33);
        WindowManager window = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        if (window != null) {
            Display display = window.getDefaultDisplay();
            float refreshRate = display.getRefreshRate();
            syncInterval = (long) (1 / refreshRate * TimeUnit.SECONDS.toMicros(1));
        }
        handler = new Handler(Looper.getMainLooper());
    }

    @Override
    public Surface getSurface() {
        synchronized (surfaceLock) {
            return surface;
        }
    }

    @Override
    public void setSurface(Surface surface) {
        synchronized (surfaceLock) {
            if (this.surface != surface) {
                this.surface = surface;
                if (surfaceChangeListener != null) {
                    surfaceChangeListener.onSurfaceChange(surface);
                }
                sizeChanged = true;
            }
        }
    }

    @Override
    public void setSurfaceChangeListener(SurfaceChangeListener listener) {
        this.surfaceChangeListener = listener;
    }

    @Override
    public void configure(MediaFormat format) {
        int width = format.getInteger(MediaFormat.KEY_WIDTH);
        int height = format.getInteger(MediaFormat.KEY_HEIGHT);

        // check output cropped
        if (format.containsKey(FORMAT_KEY_CROP_LEFT)
                && format.containsKey(FORMAT_KEY_CROP_RIGHT)
                && format.containsKey(FORMAT_KEY_CROP_TOP)
                && format.containsKey(FORMAT_KEY_CROP_BOTTOM)) {

            int cropLeft = format.getInteger(FORMAT_KEY_CROP_LEFT);
            int cropRight = format.getInteger(FORMAT_KEY_CROP_RIGHT);
            int cropTop = format.getInteger(FORMAT_KEY_CROP_TOP);
            int cropBottom = format.getInteger(FORMAT_KEY_CROP_BOTTOM);

            width = cropRight - cropLeft + 1;
            height = cropBottom - cropTop + 1;
        }

        Size current = new Size(width, height);
        if (size == null || !size.equals(current)) {
            size = current;
            sizeChanged = true;
        }
    }

    @Override
    public void flush() {
        droppedFrames = 0;
        renderedFrames = 0;
        renderPresentationTime = -1;
        lastMediaTime = -1;
        mediaTime = 0;
        startTime = 0;
    }

    @Override
    public void render(MediaCodec codec, int index, final long presentationTimeUs) {
        if (mediaTime == 0) {
            mediaTime = presentationTimeUs;
            startTime = System.nanoTime();
        }
        long now = System.nanoTime();
        long elapsed = (long) ((now - startTime) * playbackRate);
        long currentTime = mediaTime + TimeUnit.NANOSECONDS.toMicros(elapsed);
        long releaseTime = currentTime > 0 ? presentationTimeUs - currentTime : 0;
        boolean render = true;
        long renderSystemTime = 0;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP
                && releaseTime > 0 && releaseTime <= (3 * syncInterval)) {
            renderSystemTime = now + TimeUnit.MICROSECONDS.toNanos(releaseTime);
        } else {
            if (releaseTime >= -syncInterval && startTime > 0) {
                final long minRelease = TimeUnit.MILLISECONDS.toMicros(10);
                final long maxRelease = TimeUnit.SECONDS.toMicros(1);
                long sleep = releaseTime - minRelease - (2 * syncInterval);
                if (sleep > maxRelease) {
                    Log.w(Logging.TAG, "SurfaceRender waiting "
                            + sleep + " us media " + currentTime + " presentation " + presentationTimeUs);
                }
                if (sleep >= minRelease) {
                    Clock.sleep(TimeUnit.MICROSECONDS.toMillis(Math.min(sleep, maxRelease)));
                }
            } else if (renderedFrames > 0) {
                render = false;
            }

            if (render && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                renderSystemTime = System.nanoTime() + TimeUnit.MICROSECONDS.toNanos(releaseTime);
            }
        }

        synchronized (surfaceLock) {
            // handle the surface changing or released by dropping the frame
            if (surface == null) {
                render = false;
            }
            if (render && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {

                // change the render time to match the next vsync time based on the last frame callback
                long syncIntervalNanos = TimeUnit.MICROSECONDS.toNanos(syncInterval);
                long frames = (renderSystemTime - frameTimeNanos) / syncIntervalNanos;
                renderSystemTime = frameTimeNanos + (syncIntervalNanos * frames);

                codec.releaseOutputBuffer(index, renderSystemTime);
            } else {
                codec.releaseOutputBuffer(index, render);
            }
        }

        if (render) {
            renderedFrames++;
            updateVideoSize();
        } else {
            droppedFrames++;
        }
        renderPresentationTime = presentationTimeUs;
    }

    @Override
    public void render(ByteBuffer buffer, int size, long presentationTimeUs) {
        // decoder actually renders the surface, just track the rendered time
        renderPresentationTime = presentationTimeUs;
        renderedFrames++;
    }

    @Override
    public void start() {
        startTime = System.nanoTime();
        addFrameCallback();
    }

    @Override
    public void stop() {
        startTime = 0;
        mediaTime = 0;
        removeFrameCallback();
    }

    @Override
    public void release() {
        removeFrameCallback();
    }

    @Override
    public long getRenderedPresentationTime() {
        return renderPresentationTime;
    }

    @Override
    public void setMediaTime(long time) {
        if (lastMediaTime != -1 && time == -1) {
            mediaTime = 0;
            Log.d(Logging.TAG, "SurfaceRender reset time");
        }
        lastMediaTime = time;

        // periodically resync to the master media time
        if (mediaTime <= 0 || System.nanoTime() - startTime > (TimeUnit.SECONDS.toNanos(1) * playbackRate)) {
            mediaTime = time == -1 ? 0 : time;
            startTime = System.nanoTime();
        }
    }

    @Override
    public void setPlaybackRate(float rate) {
        playbackRate = rate;
    }

    @Override
    public int getDroppedFrames() {
        return droppedFrames;
    }

    @Override
    public int getRenderedFrames() {
        return renderedFrames;
    }

    @Override
    public Size getVideoSize() {
        return size;
    }

    private void updateVideoSize() {
        if (sizeChanged) {
            sizeChanged = false;
            SurfaceRenderer.VideoSizeListener listener = sizeListener.get();
            if (listener != null) {
                listener.onSizeChange(size.width, size.height);
            }
        }
    }

    // must always be called from the handler thread
    private Choreographer getChoreographer() {
        if (choreographer == null) {
            choreographer = Choreographer.getInstance();
        }
        return choreographer;
    }

    private void addFrameCallback() {
        handler.post(new Runnable() {
            @Override
            public void run() {
                getChoreographer().postFrameCallback(SurfaceRenderer.this);
            }
        });
    }

    private void removeFrameCallback() {
        handler.post(new Runnable() {
            @Override
            public void run() {
                getChoreographer().removeFrameCallback(SurfaceRenderer.this);
            }
        });
    }

    @Override
    public void doFrame(long frameTimeNanos) {
        this.frameTimeNanos = frameTimeNanos;
        getChoreographer().postFrameCallbackDelayed(this, 1000);
    }
}
