package tv.twitch.android.player;

import android.content.Context;
import android.media.MediaPlayer;
import android.net.Uri;
import android.view.Surface;

import java.io.IOException;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * Player implementation using the OS MediaPlayer. This is used for compatibility and fallback
 * use cases in which the normal player can't be used.
 *
 * @author Nikhil Purushe
 */
public class SystemPlayer implements Player {

    private Context context;
    private MediaPlayer mediaPlayer;
    private Statistics statistics;
    private final Set<Listener> listeners;
    private int percentBuffered;
    private boolean buffering;
    private boolean prepared;
    private boolean completed;
    private boolean muted;
    private float volume;

    public SystemPlayer(Context context) {
        this.context = context;
        this.listeners = new CopyOnWriteArraySet<>();
        this.volume = 1.0f;
        mediaPlayer = new MediaPlayer();
        mediaPlayer.setOnBufferingUpdateListener(new MediaPlayer.OnBufferingUpdateListener() {
            @Override
            public void onBufferingUpdate(MediaPlayer mp, int percent) {
                percentBuffered = percent;
            }
        });
        mediaPlayer.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
            @Override
            public void onPrepared(MediaPlayer mp) {
                prepared = true;
                for (Listener listener : listeners) {
                    listener.onStateChanged(getState());
                }
            }
        });
        mediaPlayer.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
            @Override
            public void onCompletion(MediaPlayer mp) {
                completed = true;
                for (Listener listener : listeners) {
                    listener.onStateChanged(getState());
                }
            }
        });
        mediaPlayer.setOnSeekCompleteListener(new MediaPlayer.OnSeekCompleteListener() {
            @Override
            public void onSeekComplete(MediaPlayer mp) {
                for (Listener listener : listeners) {
                    listener.onStateChanged(getState());
                }
            }
        });
        mediaPlayer.setOnVideoSizeChangedListener(new MediaPlayer.OnVideoSizeChangedListener() {
            @Override
            public void onVideoSizeChanged(MediaPlayer mp, int width, int height) {
                for (Listener listener : listeners) {
                    listener.onVideoSizeChanged(width, height);
                }
            }
        });
        mediaPlayer.setOnErrorListener(new MediaPlayer.OnErrorListener() {
            @Override
            public boolean onError(MediaPlayer mp, int what, int extra) {
                for (Listener listener : listeners) {
                    MediaException exception = new MediaException(MediaResult.Error, what + ":" + extra);
                    listener.onError(exception);
                }
                return true;
            }
        });
        mediaPlayer.setOnBufferingUpdateListener(new MediaPlayer.OnBufferingUpdateListener() {
            @Override
            public void onBufferingUpdate(MediaPlayer mp, int percent) {
                percentBuffered = percent;
                for (Listener listener : listeners) {
                    listener.onStateChanged(getState());
                }
            }
        });
        mediaPlayer.setOnInfoListener(new MediaPlayer.OnInfoListener() {
            @Override
            public boolean onInfo(MediaPlayer mp, int what, int extra) {
                if (what == MediaPlayer.MEDIA_INFO_BUFFERING_START) {
                    buffering = true;
                } else if (what == MediaPlayer.MEDIA_INFO_BUFFERING_END) {
                    buffering = false;
                }
                return false;
            }
        });
        statistics = new Statistics();
    }

    @Override
    public void release() {
        mediaPlayer.release();
    }

    @Override
    public int getAudioSessionId() {
        return mediaPlayer.getAudioSessionId();
    }

    @Override
    public State getState() {
        if (mediaPlayer.isPlaying()) {
            return State.PLAYING;
        } else if (completed) {
            return State.ENDED;
        } else if (prepared) {
            return State.READY;
        } else if (buffering) {
            return State.BUFFERING;
        }
        return State.IDLE;
    }

    @Override
    public long getPosition() {
        return mediaPlayer.getCurrentPosition();
    }

    @Override
    public long getDuration() {
        return mediaPlayer.getDuration();
    }

    @Override
    public long getBufferedPosition() {
        return percentBuffered * mediaPlayer.getDuration();
    }

    @Override
    public String getVersion() {
        return "1.0";
    }

    @Override
    public void addListener(Listener listener) {
        if (listener != null) {
            listeners.add(listener);
        }
    }

    @Override
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }

    @Override
    public void setLooping(boolean loop) {
        mediaPlayer.setLooping(loop);
    }

    @Override
    public boolean isMuted() {
        return muted;
    }

    @Override
    public void setMuted(boolean muted) {
        if (muted) {
            this.muted = true;
            mediaPlayer.setVolume(0, 0);
        } else {
            this.muted = false;
            mediaPlayer.setVolume(volume, volume);
        }
    }

    @Override
    public void setSurface(Surface surface) {
        mediaPlayer.setSurface(surface);
    }

    @Override
    public void setPlaybackRate(float rate) {
    }

    @Override
    public void setVolume(float volume) {
        this.volume = volume;
        mediaPlayer.setVolume(volume, volume);
    }

    @Override
    public void load(Uri uri) {
        load(uri, "");
    }

    @Override
    public void load(Uri uri, String mediaType) {
        mediaPlayer.reset();
        prepared = false;
        completed = false;
        try {
            mediaPlayer.setDataSource(context, uri);
            mediaPlayer.prepareAsync();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public boolean isSeekable() {
        return true;
    }

    @Override
    public void seekTo(long position) {
        mediaPlayer.seekTo((int) position);
    }

    @Override
    public void play() {
        mediaPlayer.start();
    }

    @Override
    public void pause() {
        mediaPlayer.pause();
    }

    @Override
    public Set<Quality> getQualities() {
        return Collections.emptySet();
    }

    @Override
    public Quality getQuality() {
        return null;
    }

    @Override
    public void setQuality(Quality quality) {
    }

    @Override
    public void setQuality(Quality quality, boolean adaptive) {
    }

    @Override
    public boolean getAutoSwitchQuality() {
        return true;
    }

    @Override
    public void setAutoSwitchQuality(boolean enable) {
    }

    @Override
    public void setAutoInitialBitrate(int bitrate) {
    }

    @Override
    public void setAutoMaxBitrate(int bitrate) {
    }

    @Override
    public void setAutoMaxVideoSize(int width, int height) {
    }

    @Override
    public void setLiveLowLatencyEnabled(boolean enabled) {
    }

    @Override
    public void setMinBuffer(int durationMs) {
    }

    @Override
    public void setMaxBuffer(int durationMs) {
    }

    @Override
    public void setClientId(String id) {
    }

    @Override
    public void setDeviceId(String id) {
    }

    @Override
    public long getAverageBitrate() {
        return 0;
    }

    @Override
    public long getBandwidthEstimate() {
        return -1;
    }

    @Override
    public long getLiveLatency() {
        return -1;
    }

    @Override
    public Statistics getStatistics() {
        return statistics;
    }
}
