package tv.twitch.android.player;

import android.content.Context;
import android.media.MediaCodec;
import android.media.MediaCodecInfo;
import android.media.MediaCodecList;
import android.media.MediaFormat;
import android.os.Build;
import android.util.Log;
import android.view.Display;
import android.view.WindowManager;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * VideoCapabilities provides apis for determining the video decoding capabilities of the current
 * device for different types of video codecs.
 */
public class VideoCapabilities {

    int maxBitrate;
    int maxFramerate;
    int maxWidth;
    int maxHeight;
    int maxProfile;
    int maxLevel;
    int maxInstances;

    private static final Map<String, VideoCapabilities> CAPABILITIES = new HashMap<>();
    private static final List<String> VP9_SOFTWARE_CODECS = Collections.singletonList(MediaCodecFactory.OMX_SOFTWARE_VP9_DECODER);

    private VideoCapabilities() {
    }

    /**
     * @return true if VP9 is supported in hardware for the current device, false otherwise.
     */
    public static boolean isVP9Supported() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            MediaCodecList list = new MediaCodecList(MediaCodecList.REGULAR_CODECS);
            MediaFormat format = MediaFormat.createVideoFormat(MediaType.VIDEO_VP9, 1920, 1080);
            String decoder = list.findDecoderForFormat(format);
            // ignore software VP9 decoder
            return !VP9_SOFTWARE_CODECS.contains(decoder);
        }
        return false;
    }

    /**
     * Get the codec capabilities for a specific media type {@link MediaType}.
     *
     * @param context application context
     * @param mediaType media type of the video e.g. {@link MediaType#VIDEO_AVC}
     * @return capabilities of the given codec
     */
    public static VideoCapabilities getCapabilities(Context context, String mediaType) {
        if (CAPABILITIES.containsKey(mediaType)) {
            return CAPABILITIES.get(mediaType);
        }

        VideoCapabilities capabilities = new VideoCapabilities();
        capabilities.maxBitrate = Integer.MAX_VALUE;
        capabilities.maxWidth = 1920;
        capabilities.maxHeight = 1280;
        capabilities.maxFramerate = 30;
        capabilities.maxProfile = 100;
        capabilities.maxLevel = 41;
        capabilities.maxInstances = 1;

        // lollipop try to find the decoder info in the codec list
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            try {
                List<MediaCodecInfo> info = MediaCodecFactory.findSupportedCodecs(mediaType, false);

                if (!info.isEmpty()) {
                    capabilities.setVideoCapabilities(mediaType, info.get(0));
                    return capabilities;
                }
            } catch (Exception e) {
                Log.e(Logging.TAG, "Failed to get codec capabilities", e);
            }
        } else if (context != null) {
            // Android < 5.0 cap the max video size to the display
            WindowManager window = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
            if (window != null) {
                Display display = window.getDefaultDisplay();
                capabilities.maxWidth = Math.min(capabilities.maxWidth, display.getWidth());
                capabilities.maxHeight = Math.min(capabilities.maxHeight, display.getHeight());
            }
        }

        // create the decoder to get it's capabilities
        MediaCodec codec = null;
        try {
            codec = MediaCodecFactory.createCodec(mediaType, false);
            if (codec != null) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
                    capabilities.setVideoCapabilities(mediaType, codec.getCodecInfo());
                }
            }
        } catch (Exception e) {
            Log.e(Logging.TAG, "Failed to get codec capabilities", e);
        } finally {
            if (codec != null) {
                codec.release();
            }
        }

        // cache the capabilities to void unnecessary creation of codec instances
        CAPABILITIES.put(mediaType, capabilities);
        return capabilities;
    }

    private void setVideoCapabilities(String mediaType, MediaCodecInfo info) {
        MediaCodecInfo.CodecCapabilities codecCaps = info.getCapabilitiesForType(mediaType);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            MediaCodecInfo.VideoCapabilities video = codecCaps.getVideoCapabilities();
            maxBitrate = video.getBitrateRange().getUpper();
            maxWidth = video.getSupportedWidths().getUpper();
            maxHeight = video.getSupportedHeights().getUpper();
            maxFramerate = video.getSupportedFrameRates().getUpper();
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                maxInstances = codecCaps.getMaxSupportedInstances();
            }
        }

        if (MediaCodecFactory.limitMaxDecodeSize(info.getName())) {
            maxWidth = 1920;
            maxHeight = 1088;
        }

        int maxProfile = 0;
        int maxLevel = 0;

        for (int i = 0; i < codecCaps.profileLevels.length; i++) {
            int profile = codecCaps.profileLevels[i].profile;
            int level = codecCaps.profileLevels[i].level;

            if (profile > maxProfile) {
                if (mediaType.equals(MediaType.VIDEO_AVC)) {
                    if (profile <= MediaCodecInfo.CodecProfileLevel.AVCProfileHigh444) {
                        maxProfile = profile;
                    }
                } else {
                    maxProfile = profile;
                }
            }
            if (level > maxLevel) {
                maxLevel = level;
            }
        }

        if (mediaType.equals(MediaType.VIDEO_AVC)) {
            if (maxProfile == 0) {
                maxProfile = MediaCodecInfo.CodecProfileLevel.AVCProfileHigh;
            }
            if (maxLevel == 0) {
                maxLevel = MediaCodecInfo.CodecProfileLevel.AVCLevel42;
            }
        } else if (mediaType.equals(MediaType.VIDEO_VP9)) {
            if (maxProfile == 0) {
                maxProfile = 0x01; // MediaCodecInfo.CodecProfileLevel.VP9Profile0;
            }
            if (maxLevel == 0) {
                maxLevel = 0x80; // MediaCodecInfo.CodecProfileLevel.VP9Level41;
            }
        }

        if (mediaType.equals(MediaType.VIDEO_AVC)) {
            this.maxProfile = avcProfileIndicationFromOMX(maxProfile);
            this.maxLevel = avcLevelIndicationFromOMX(maxLevel);
        } else if (mediaType.equals(MediaType.VIDEO_VP9)) {
            this.maxProfile = vp9ProfileIndicationFromOMX(maxProfile);
            this.maxLevel = vp9LevelIndicationFromOMX(maxLevel);
        }
    }

    private static int avcProfileIndicationFromOMX(int profile) {
        switch (profile) {
            case MediaCodecInfo.CodecProfileLevel.AVCProfileBaseline:
                return 66;
            default:
            case MediaCodecInfo.CodecProfileLevel.AVCProfileMain:
                return 77;
            case MediaCodecInfo.CodecProfileLevel.AVCProfileExtended:
                return 88;
            case MediaCodecInfo.CodecProfileLevel.AVCProfileHigh:
                return 100;
            case MediaCodecInfo.CodecProfileLevel.AVCProfileHigh10:
                return 110;
            case MediaCodecInfo.CodecProfileLevel.AVCProfileHigh422:
                return 122;
            case MediaCodecInfo.CodecProfileLevel.AVCProfileHigh444:
                return 244;
        }
    }

    private static int avcLevelIndicationFromOMX(int level) {
        switch (level) {
            case MediaCodecInfo.CodecProfileLevel.AVCLevel1b:
                return 9;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel1:
                return 10;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel11:
                return 11;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel12:
                return 12;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel13:
                return 13;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel2:
                return 20;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel21:
                return 21;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel22:
                return 22;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel3:
                return 30;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel31:
                return 31;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel32:
                return 32;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel4:
                return 40;
            default:
            case MediaCodecInfo.CodecProfileLevel.AVCLevel41:
                return 41;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel42:
                return 42;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel5:
                return 50;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel51:
                return 51;
            case MediaCodecInfo.CodecProfileLevel.AVCLevel52:
                return 52;
        }
    }

    private static int vp9ProfileIndicationFromOMX(int profile) {
        switch (profile) {
            default:
            case MediaCodecInfo.CodecProfileLevel.VP9Profile0:
                return 0;
            case MediaCodecInfo.CodecProfileLevel.VP9Profile1:
                return 1;
            case MediaCodecInfo.CodecProfileLevel.VP9Profile2:
            case MediaCodecInfo.CodecProfileLevel.VP9Profile2HDR:
                return 2;
            case MediaCodecInfo.CodecProfileLevel.VP9Profile3:
            case MediaCodecInfo.CodecProfileLevel.VP9Profile3HDR:
                return 3;
        }
    }

    private static int vp9LevelIndicationFromOMX(int level) {
        switch (level) {
            case MediaCodecInfo.CodecProfileLevel.VP9Level1:
                return 10;
            case MediaCodecInfo.CodecProfileLevel.VP9Level2:
                return 20;
            case MediaCodecInfo.CodecProfileLevel.VP9Level3:
                return 30;
            case MediaCodecInfo.CodecProfileLevel.VP9Level4:
                return 40;
            case MediaCodecInfo.CodecProfileLevel.VP9Level5:
                return 50;
            case MediaCodecInfo.CodecProfileLevel.VP9Level6:
                return 60;
            case MediaCodecInfo.CodecProfileLevel.VP9Level11:
                return 11;
            case MediaCodecInfo.CodecProfileLevel.VP9Level21:
                return 21;
            case MediaCodecInfo.CodecProfileLevel.VP9Level31:
                return 31;
            case MediaCodecInfo.CodecProfileLevel.VP9Level41:
                return 41;
            default:
            case MediaCodecInfo.CodecProfileLevel.VP9Level51:
                return 51;
            case MediaCodecInfo.CodecProfileLevel.VP9Level61:
                return 61;
            case MediaCodecInfo.CodecProfileLevel.VP9Level62:
                return 62;
        }
    }

    /**
     * @return max bitrate of the video codec.
     */
    public int getMaxBitrate() {
        return maxBitrate;
    }

    /**
     * @return max supported video frame rate.
     */
    public int getMaxFramerate() {
        return maxFramerate;
    }

    /**
     * @return max decodable video width.
     */
    public int getMaxWidth() {
        return maxWidth;
    }

    /**
     * @return max decodable video height.
     */
    public int getMaxHeight() {
        return maxHeight;
    }

    /**
     * @return max supported profile.
     */
    public int getMaxProfile() {
        return maxProfile;
    }

    /**
     * @return max supported level.
     */
    public int getMaxLevel() {
        return maxLevel;
    }

    /**
     * @return max number of concurrent decoder instances supported in hardware.
     */
    public int getMaxInstances() {
        return maxInstances;
    }

    @Override
    public String toString() {
        return "maxBitrate=" + maxBitrate +
                ", maxFramerate=" + maxFramerate +
                ", maxWidth=" + maxWidth +
                ", maxHeight=" + maxHeight +
                ", maxProfile=" + maxProfile +
                ", maxLevel=" + maxLevel;
    }
}
