package tv.twitch.android.player;

import android.graphics.Matrix;
import android.view.SurfaceView;
import android.view.TextureView;
import android.view.View;
import android.view.ViewGroup;

/**
 * Helper class for resizing views to fit a given video dimensions.
 */
public class ViewUtil {

    private ViewUtil() {}

    /**
     * Updates a {@link SurfaceView}'s layout params to fit the given video dimensions
     * @param surfaceView view to set the transform on
     * @param width video width
     * @param height video height
     */
    public static void setLayoutParams(SurfaceView surfaceView, int width, int height) {
        View parent = (View) surfaceView.getParent();
        Size scaled = scaleTo(parent, width, height);
        ViewGroup.LayoutParams layoutParams = surfaceView.getLayoutParams();
        layoutParams.width = scaled.width;
        layoutParams.height = scaled.height;
        surfaceView.setLayoutParams(layoutParams);
    }

    /**
     * Updates a {@link TextureView}'s transform to fit the given video dimensions
     * @param textureView view to set the transform on
     * @param width video width
     * @param height video height
     */
    public static void setTransform(TextureView textureView, int width, int height) {
        int viewWidth = textureView.getWidth();
        int viewHeight = textureView.getHeight();
        Size scaled = scaleTo(textureView, width, height);
        Matrix matrix = new Matrix();
        textureView.getTransform(matrix);
        float scaleX = (float) scaled.width / viewWidth;
        float scaleY = (float) scaled.height / viewHeight;
        matrix.setScale(scaleX, scaleY);
        int x = (viewWidth - scaled.width) / 2;
        int y = (viewHeight - scaled.height) / 2;
        matrix.postTranslate(x, y);
        textureView.setTransform(matrix);
    }

    private static Size scaleTo(View view, int width, int height) {
        int viewWidth = view.getWidth();
        int viewHeight = view.getHeight();
        float ratio = height / (float) width;
        int scaledWidth;
        int scaledHeight;
        if (viewHeight > viewWidth * ratio) {
            scaledWidth = viewWidth;
            scaledHeight = (int) (viewWidth * ratio);
        } else {
            scaledWidth = (int) (viewHeight / ratio);
            scaledHeight = viewHeight;
        }
        return new Size(scaledWidth, scaledHeight);
    }
}
