package tv.twitch.android.player.http;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

/**
 * Executes an http request using {@link HttpURLConnection}.
 *
 * @author Nikhil Purushe
 */
@SuppressWarnings("unused") // called from native
public class HttpUrlConnectionClient implements Client {

    private static final int DEFAULT_CONNECT_TIMEOUT_MS = 10000;
    private static final int DEFAULT_READ_TIMEOUT_MS = 10000;
    private static final int BUFFER_SIZE = 16 * 1024;
    private final ExecutorService executor = Executors.newFixedThreadPool(3);

    HttpUrlConnectionClient() {
    }

    @Override
    public void release() {
        executor.shutdown();
    }

    @Override
    public void execute(Request request, ResponseCallback callback) {
        executeAsync(request, callback);
    }

    private void executeAsync(final Request request, final ResponseCallback callback) {
        Future<?> future = executor.submit(new Runnable() {
            @Override
            public void run() {
                executeSync(request, callback);
            }
        });
        request.setFuture(future);
    }

    private void executeSync(final Request request, ResponseCallback callback) {
        HttpURLConnection connection = null;
        try {
            connection = openConnection(request);
            Response response = blockingExecute(connection, request, callback);
            synchronized (request.lock()) {
                if (!request.isCancelled()) {
                    callback.onResponse(response);
                }
            }

        } catch (IOException e) {
            disconnect(connection);
            e.printStackTrace();
            synchronized (request.lock()) {
                if (!request.isCancelled()) {
                    callback.onError(e);
                }
            }
        }
    }

    private HttpURLConnection openConnection(Request request) throws IOException {
        final URL url = new URL(request.getUrl());
        return (HttpURLConnection) url.openConnection();
    }

    private Response blockingExecute(HttpURLConnection connection, Request request, ResponseCallback callback) throws IOException {
        String method = request.getMethod().toString().toUpperCase(Locale.ROOT);
        connection.setRequestMethod(method);
        connection.setInstanceFollowRedirects(true);
        if (request.getTimeout() > 0) {
            connection.setConnectTimeout((int) TimeUnit.SECONDS.toMillis(request.getTimeout()));
            connection.setReadTimeout(DEFAULT_READ_TIMEOUT_MS);
        } else {
            connection.setConnectTimeout(DEFAULT_CONNECT_TIMEOUT_MS);
            connection.setReadTimeout(DEFAULT_READ_TIMEOUT_MS);
        }

        for (Map.Entry<String, String> header : request.getHeaders().entrySet()) {
            connection.setRequestProperty(header.getKey(), header.getValue());
        }
        if (request.getContent() != null) {
            writeRequestContent(connection, request.getContent(), callback);
        }
        connection.connect();
        final int statusCode = connection.getResponseCode();

        final Response response = new Response(statusCode);
        for (Map.Entry<String, List<String>> header : connection.getHeaderFields().entrySet()) {
            List<String> values = header.getValue();
            response.getHeaders().put(header.getKey(), values.get(0));
        }

        // handle redirect
        if (statusCode == HttpURLConnection.HTTP_MOVED_TEMP ||
            statusCode == HttpURLConnection.HTTP_MOVED_PERM) {
            String location = response.getHeaders().get("Location");
            if (location != null) {
                Request redirect = new Request(location, request.getMethod());
                redirect.getHeaders().putAll(request.getHeaders());
                disconnect(connection);
                return blockingExecute(openConnection(redirect), redirect, callback);
            }
        }
        response.setConsumer(new ResponseStreamConsumer(request, response, connection));
        return response;
    }

    private static void close(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private static void disconnect(HttpURLConnection connection) {
        if (connection != null) {
            try {
                connection.disconnect();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private void writeRequestContent(HttpURLConnection connection, ByteBuffer content, ResponseCallback callback)
            throws IOException {

        connection.setDoOutput(true);
        OutputStream outputStream = connection.getOutputStream();

        try {
            WritableByteChannel channel = Channels.newChannel(outputStream);
            channel.write(content);
        } catch (IOException e) {
            e.printStackTrace();
            callback.onError(e);
        } finally {
            close(outputStream);
        }
    }

    private static class ResponseStreamConsumer implements StreamConsumer {

        private final Request request;
        private final Response response;
        private final HttpURLConnection connection;

        ResponseStreamConsumer(Request request, Response response, HttpURLConnection connection) {
            this.request = request;
            this.response = response;
            this.connection = connection;
        }

        @Override
        public void consume(ReadCallback callback) {
            InputStream inputStream = null;
            IOException ioException = null;
            if (callback.getTimeout() > 0) {
                connection.setReadTimeout((int) TimeUnit.SECONDS.toMillis(callback.getTimeout()));
            }
            try {
                inputStream = connection.getErrorStream();
                if (inputStream == null) {
                    inputStream = connection.getInputStream();
                }
                ByteBuffer buffer = ByteBuffer.allocateDirect(BUFFER_SIZE);
                byte[] bytes = new byte[BUFFER_SIZE];
                boolean endOfStream = false;

                while (!endOfStream && !request.isCancelled()) {
                    int read = inputStream.read(bytes);
                    int size = 0;
                    buffer.rewind();
                    if (read == -1) {
                        endOfStream = true;
                    } else {
                        buffer.put(bytes, 0, read);
                        size = read;
                    }
                    callback.onBuffer(buffer, size, endOfStream);
                }
            } catch (IOException e) {
                ioException = e;
                e.printStackTrace();
            } finally {
                close(inputStream);
                disconnect(connection);
            }

            if (ioException != null) {
                callback.onError(ioException);
            }
        }
    }
}
