package tv.twitch.android.player.http;

import java.nio.ByteBuffer;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Future;

@SuppressWarnings("unused") // called from native
public class Request {

    private final String url;
    private Method method;
    private final Map<String, String> headers = new LinkedHashMap<>();
    private WriteCallback streamCallback;
    private ByteBuffer content;
    private int timeoutSeconds = 10;
    private boolean cancelled;
    private Future<?> future;

    public Request(String url, Method method) {
        this.url = url;
        this.method = method;
    }

    public Request(String url, String method) {
        this.url = url;
        try {
            this.method = Method.valueOf(method.toUpperCase());
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            this.method = Method.GET;
        }
    }

    public String getUrl() {
        return url;
    }

    public Method getMethod() {
        return method;
    }

    public ByteBuffer getContent() {
        return content;
    }

    public void setContent(ByteBuffer content) {
        // passed in ByteBuffer only has stack scope so need a copy
        ByteBuffer copy = ByteBuffer.allocateDirect(content.remaining());
        copy.put(content);
        copy.rewind();
        this.content = copy;
    }

    public WriteCallback getContentCallback() {
        return streamCallback;
    }

    public void setContentCallback(WriteCallback callback) {
        this.streamCallback = callback;
    }

    public void setHeader(String key, String value) {
        getHeaders().put(key, value);
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public int getTimeout() {
        return timeoutSeconds;
    }

    public void setTimeout(int seconds) {
        this.timeoutSeconds = seconds;
    }

    void cancel() {
        synchronized (lock()) {
            cancelled = true;
            if (future != null) {
                future.cancel(true);
            }
        }
    }

    boolean isCancelled() {
        synchronized (lock()) {
            return cancelled;
        }
    }

    public void setFuture(Future<?> future) {
        synchronized (lock()) {
            this.future = future;
        }
    }

    Object lock() {
        return this;
    }
}
