#import "AVPlayerSink.h"
#include "media/MediaReader.hpp"
#include <thread>

namespace twitch {
AVPlayerSink::AVPlayerSink(MediaSink::Listener& listener, std::shared_ptr<Scheduler> scheduler)
    : ScopedScheduler(scheduler)
    , m_listener(listener)
    , m_scheduler(scheduler)
    , m_paused(true)
    , m_sequenceNum(1)
    , m_ended(false)
    , m_format(nullptr)
    , m_isPassthrough(false)
{
    m_player = [AVPlayer playerWithPlayerItem:nil];
    // Create observer to keep track of player status
    m_observer = [[TTVAVPlayerObserver alloc] initWithPlayer:m_player:&m_listener:m_scheduler:this];
}

AVPlayerSink::~AVPlayerSink()
{
    m_observer = nil;
}

bool AVPlayerSink::isAirPlayEnabled()
{
    AVAudioSession* audioSession = [AVAudioSession sharedInstance];
    AVAudioSessionRouteDescription* currentRoute = audioSession.currentRoute;
    for (AVAudioSessionPortDescription* outputPort in currentRoute.outputs) {
        if ([outputPort.portType isEqualToString:AVAudioSessionPortAirPlay]) {
            return true;
        }
    }
    return false;
}

void AVPlayerSink::configure(int track, std::shared_ptr<MediaFormat> format)
{
    (void)track;

    if (format) {
        m_format = format;
        const auto& path = format->getPath();
        const auto& mediaType = format->getType();
        m_isPassthrough = (mediaType == MediaType::Video_MP4 && path.find(".mp4") != std::string::npos);
        if (m_isPassthrough) {
            init();
        }
        if (m_observer != nil) {
            [m_observer enqueueFormat:m_format];
        }
    }
}

void AVPlayerSink::enqueue(int track, std::shared_ptr<MediaSampleBuffer> sample)
{
    if (track != media::MediaReader::MetaTrackId) {
        if (sample->type == media::FragmentSample::Type) {
            auto segment = std::static_pointer_cast<media::FragmentSample>(sample);
            std::unique_lock<Mutex> lock(m_mutex);

            m_segments[m_sequenceNum].emplace_back(segment);
            if (segment->isEndOfStream) {
                m_sequenceNum++;
                // erase old cached segment data
                const int MaxSegments = 30;
                if (m_segments.size() > MaxSegments) {
                    for (int i = m_segments.begin()->first; i < m_sequenceNum - MaxSegments; i++) {
                        m_segments.erase(i);
                    }
                }
            }

            if (m_sequenceNum == 2 && segment->isEndOfStream) {
                init();
                //enqueue first format
                [m_observer enqueueFormat:m_format];
            }
        } else {
            m_listener.onSinkError(Error(ErrorSource::Render, MediaResult::ErrorInvalidParameter, "invalid sample"));
        }
    } else if (m_observer != nil) {
        [m_observer enqueueSample:sample];
    }
}

std::vector<uint8_t> AVPlayerSink::readSegmentData(int sequence)
{
    std::unique_lock<Mutex> lock(m_mutex);

    if (sequence > m_sequenceNum) {
        return std::vector<uint8_t>();
    }

    size_t index = m_currentIndexes[sequence];
    auto& samples = getSamples(sequence);
    for (size_t i = index; i < samples.size(); i++) {
        auto& sample = samples[i];
        if (sample->buffer.empty() && sample->isEndOfStream) {
            m_currentIndexes.erase(sequence);
            return sample->buffer;
        }
        m_currentIndexes[sequence] = i + 1;
        return sample->buffer;
    }
    if (index >= samples.size() && !samples.empty() && samples.back()->isEndOfStream) {
        m_currentIndexes.erase(sequence); // end of segment
    }
    return std::vector<uint8_t>();
}

void AVPlayerSink::init()
{
    NSURL* url;
    AVURLAsset* asset;

    bool airplayEnabled = m_player.allowsExternalPlayback && isAirPlayEnabled();

    NSString* urlString = m_format ? [NSString stringWithCString:m_format->getPath().c_str() encoding:[NSString defaultCStringEncoding]] : @"";
    if (m_isPassthrough || airplayEnabled) {
        url = [NSURL URLWithString:urlString];
        asset = [AVURLAsset URLAssetWithURL:url options:nil];
    } else {
        NSString* playlistURL = [NSString stringWithFormat:@"twitch://%@_playlist.m3u8", urlString];
        url = [NSURL URLWithString:playlistURL];
        asset = [AVURLAsset URLAssetWithURL:url options:nil];

        // Create webserver and setup delegate for custom URL.
        if (m_delegate == nil) {
            m_delegate = [[TTVLoopbackResourceLoaderDelegate alloc] init:this];
        }
        AVAssetResourceLoader* resourceLoader = asset.resourceLoader;
        [resourceLoader setDelegate:m_delegate queue:dispatch_queue_create("AVARLDelegate loader", nil)];
    }

    AVPlayerItem* item = [AVPlayerItem playerItemWithAsset:asset];
    NSURL* newURL = [(AVURLAsset *)item.asset URL];
    NSURL* oldURL = [(AVURLAsset *)m_player.currentItem.asset URL];

    if (![[oldURL absoluteString] isEqualToString:[newURL absoluteString]]) {
        [m_player replaceCurrentItemWithPlayerItem:item];
    } else {
        AVPlayerItem* item = m_player.currentItem;
        [m_player replaceCurrentItemWithPlayerItem:nil];
        [m_player replaceCurrentItemWithPlayerItem:item];
    }

    if (airplayEnabled && m_paused) {
        [m_player play];
    }
}

void AVPlayerSink::reset()
{
    std::unique_lock<Mutex> lock(m_mutex);

    m_paused = true;
    m_segments.clear();
    m_currentIndexes.clear();
    m_sequenceNum = 1;
    //[m_player replaceCurrentItemWithPlayerItem:nil];
    if (m_delegate != nil) {
        if (m_delegate->server.isRunning) {
            [m_delegate->server stop];
        }
        m_delegate = nil;
    }
}

void AVPlayerSink::prepare()
{
}

void AVPlayerSink::play()
{
    if (m_paused) {
        m_paused = false;
        [m_player play];
    }
}

void AVPlayerSink::pause()
{
    m_paused = true;
    [m_player pause];
}

void AVPlayerSink::seekTo(MediaTime time)
{
    CMTime cmtime = CMTimeMake(time.count(), time.timebase());
    if (m_isPassthrough) {
        [m_player seekToTime:cmtime];
    } else {
        [m_player pause];
        std::unique_lock<Mutex> lock(m_mutex);
        m_segments.clear();
        m_currentIndexes.clear();
        m_sequenceNum = 1;
        [m_observer seekTo:cmtime];
    }
}

void AVPlayerSink::setPlaybackRate(float rate)
{
    [m_player setRate:rate];
}

void AVPlayerSink::setSurface(void* surface)
{
    AVPlayerLayer* layer = (__bridge AVPlayerLayer*)(surface);
    if (layer != nil) {
        layer.player = m_player;
    }
}

void AVPlayerSink::setVolume(float volume)
{
    [m_player setVolume:volume];
}

const std::vector<std::shared_ptr<media::FragmentSample>>& AVPlayerSink::getSamples(int sequence)
{
    return m_segments[sequence];
}

std::string AVPlayerSink::generatePlaylist(const std::string& baseUrl)
{
    std::unique_lock<Mutex> lock(m_mutex);
    hls::MediaPlaylist playlist;
    playlist.setVersion(6);
    playlist.setTargetDuration(2);
    playlist.setEndOfStream(m_ended);

    for (const auto& entry : m_segments) {
        if (!entry.second.empty()) {
            const auto& sample = entry.second.at(0);
            if (!entry.second.back()->isEndOfStream) { // enables only completed segments
                break;
            }

            hls::Segment segment;
            MediaTime duration = sample->mediaDuration != MediaTime::zero() ? sample->mediaDuration : sample->duration;
            if (duration.seconds() > playlist.getTargetDuration()) {
                playlist.setTargetDuration(duration.seconds());
            }
            segment.duration = duration;
            segment.discontinuity = sample->isDiscontinuity;
            segment.sequenceNumber = entry.first;
            segment.isInitialization = sample->isInitialization;

            std::string extension;
            if (m_format) {
                if (m_format->getType().matches(MediaType::Video_MP2T)) {
                    extension = ".ts";
                } else if (m_format->getType().matches(MediaType::Video_MP4)) {
                    extension = ".mp4";
                }
            }
            segment.url = baseUrl + std::to_string(segment.sequenceNumber) + extension;
            playlist.segments().emplace_back(std::move(segment));
        }
    }

    return playlist.generate();
}
}
